{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Builtin (
	nativeBuiltins, substituteLibraryBuiltins,
	types_of_native_builtins,
	list_anyterm) where
import Maybe
import List
import IO (Handle, hPutStr, hPutStrLn, stdout, stderr)
import Data.Map hiding (map, lookup, partition, filter)
import Data.IORef
import Control.Monad.Identity
import Debug.Trace

import Basic
import Position
import Kind
import Type hiding (prop, nameT, functionT, cofunctionT, forallT, existsT, stringT, booleanT, andT, falseT, iOVT, iONT, iOT_V, iOT_N, notT, listT, integerT, trueT)
import qualified Type
import AST
import Intermediate
import Lower

namePattern nm = Pattern noT $ NamePat nm
nameCopattern nm = Copattern noT $ NameCopat nm
specificPattern stp pat = Pattern noT $ SpecificPat (TypeExpression stp) pat
packageCopattern stp copat = Copattern noT $ PackageCopat (TypeExpression stp) copat
prop = Type.prop no_span
functionT = Type.functionT (UseLowerFunctionTypes False) no_span
cofunctionT = Type.cofunctionT (UseLowerFunctionTypes False) no_span
nameT = Type.nameT no_span
forallT = Type.forallT no_span
existsT = Type.existsT no_span
stringT = Type.stringT no_span
booleanT = Type.booleanT no_span
andT = Type.andT no_span PlainOrigin
falseT = Type.falseT no_span
trueT = Type.trueT no_span
iOVT = Type.iOVT (UseLowerIOType False) no_span
iONTnot mm uio tp = if sm then (if lio then notT (iOT_N mm tp)
				 else error "--lt with --sm")
			else Type.iONT uio no_span (notT tp)
  where
    UseStrictMonad sm = mm
    UseLowerIOType lio = uio
iONT = Type.iONT (UseLowerIOType False) no_span
iOT_V = Type.iOT_V no_span
iOT_N mm = Type.iOT_N mm no_span
notT = Type.notT no_span
listT = Type.listT no_span
integerT = Type.integerT no_span
proposition = Type Proposition no_span NoT

unknown_stack_size = Nothing
predicted_stack_size n = Just n

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
instance Anyterm_ Term_ where
    term_to_any = id
    coterm_to_any = Plan_
    oblivious_anyterm stmt = Coabstract_ "@a" stmt unknown_stack_size

    -- function_anyterm:: ((Stack -> Term_) -> Stack -> Term_) -> String -> Stack -> Term_
    function_anyterm f name stack =
	    runIdentity $ byValueFunction "function anyterm" name (f' {- (Variable -1) -}) stack
	where
	    -- f' creates a Term_ suitable for byValueFunction, referring to the parameter
	    -- at various offsets as needed
	    -- stk is as received from byValueFunction, in need of a push.
	    -- We don't name the variable, because there's no good way to avoid name conflicts.
	    f' stk = return $ f (var_at var_location) stk'
		where
		    stk' = push_gapvariable stk
		    var_location = -1 - stack_size stk

    -- io_anyterm :: String -> Type -> IO Term_ -> Term_
    io_anyterm _ name tpout iot =
		    -- <iot>do:IO t      iot:IO t
		    Do_ "io_anyterm" name iot
    useList = useBuiltin_ (\term -> case term of
	    BuiltinList terms -> terms
	    otherwise -> error "list term required")
    list_anyterm = Builtin_ . BuiltinList
    be _ fterm stk = Be_ "be" (Use_ (fterm stk))

    -- lambda_anyterm :: (Term_ -> Term_) -> String -> Term_
    lambda_anyterm f name = action_term (noT,noT)
	where
	    -- If the function takes a Term of type tpin to a Term for type tpout,
	    -- then the action_term has type orT [notT tpin, [notT (notT tpout)].
	    action_term :: (Type, Type) -> Term_
	    action_term (tpin, tpout) =
		    -- [z.(z * Function_ \t -> use<f t>)]pln    z: s&~t
		    Plan_ $ Abstract_ "@z" (Cut_ "lambda_anyterm Term_"
			    (Variable_ {-z-} 0)
			    (Function_ "lambda_anyterm" name (\term ->(Use_ (f term))))
		    ) unknown_stack_size

    which_way = const ByValue

    -- t -> String
    primitive_dump = lambda_anyterm (term_to_any . (\val -> Builtin_ (BuiltinString $ old_show [] val))) "dump"

-- So that native_builtins can work with both terms and coterms:
class Anyterm_ t where
    term_to_any :: Term_ -> t
    coterm_to_any :: Coterm_ -> t
    oblivious_anyterm :: Statement_ -> t

    io_anyterm :: MonadModel -> String -> Type -> IO Term_ -> t

    -- This is a limited-use utility, because the Int offset is the index which will be
    -- given to every occurrence of the variable. For example, a very simple function
    -- would have an offset of -1 because the variable would reference the immediately
    -- containing abstraction.
    function_anyterm :: ((Stack -> t) -> Stack -> t) -> String -> Stack -> t
    -- was function_anyterm :: (t -> t) -> String -> t

    useList :: Term_ -> [t]
    list_anyterm :: [t] -> t
    be :: MonadModel -> (Stack -> t) -> Stack -> t
    lambda_anyterm :: (Term_ -> t) -> String -> t
    which_way :: t -> Way
    primitive_dump :: t  -- a -> String

instance Anyterm_ Coterm_ where
    term_to_any = Use_
    coterm_to_any = id
    oblivious_anyterm stmt = Abstract_ "@x" stmt unknown_stack_size

    -- function_anyterm :: ((Stack -> Coterm_) -> Stack -> Coterm_) -> String -> Stack -> Coterm_
    function_anyterm f name stk = 
	    runIdentity $ byNameFunction "function_anyterm" ('_':name) f' {- (Covariable_ -1) -} stk
	where
	    -- f' creates a Coterm_ suitable for byNameFunction, referring to the parameter
	    -- at various offsets as needed
	    -- stk is as received from byNameFunction, in need of a push.
	    -- We don't name the covariable, because there's no good way to avoid name conflicts.
	    f' stk = return $ f (covar_at covar_location) stk'
		where
		    stk' = push_gapvariable stk
		    covar_location = -1 - stack_size stk

    -- io_anyterm :: String -> Type -> IO Term_ -> Coterm_
    io_anyterm (UseStrictMonad strict_monad) name tpout iot =
	    -- use<<iot>do>:~!t
	    -- use<do<fmap (\y ->[use<y>]pln) iot>>:~!~~t
	    Use_ $ Do_ "io_anyterm" ('_':name) (do
					trm <- iot
					return $ if strict_monad then trm else Plan_ (Use_ trm)
	    )

    useList t = case t of
	    Builtin_ b@(BuiltinColist coterms) -> coterms
	    otherwise -> error "colist term required"
    list_anyterm = error "colist not implemented"
    be (UseStrictMonad strict_monad) fcoterm stk = Use_ (Be_ "be" ((if strict_monad then id else Use_ . Plan_) (fcoterm stk)))

    -- lambda_anyterm :: (Term_ -> Coterm_) -> String -> Coterm_
    lambda_anyterm f name = action_coterm
    -- f takes a term of type tpin(s) to a coterm of type tpout(~t).
	where
	    --  z.(z * fst[use<[x.(z * snd[w.(<x,w>par * Function_ f)])]pln>]):~~s&~t  x:s  w:~t  Function_:s&~t
	    action_coterm :: Coterm_
	    action_coterm  =
		    Abstract_ "@z" (Cut_ "lambda_anyterm Coterm_"
			(Variable_ {-z-} 0)
			(Selector_ 2 argument (Use_ (Plan_
			    (Abstract_ "@x" (Cut_ "lambda_anyterm Coterm_"
				(Variable_ {-z-} 1)
				(Selector_ 2 continuation (Abstract_ "@w" (Cut_ "lambda_anyterm Coterm_"
				    (Record_ [Variable_ {-x-} 1, Variable_ {-w-} 0])
				    (Function_ "lambda_anyterm" ('_':name) f)
				) unknown_stack_size))
			    ) unknown_stack_size)
			)))
		    ) unknown_stack_size
	    (continuation, argument) = (1, 0)

    which_way = const ByName

    -- a -> String
    primitive_dump = action_coterm
	where
	    --  z.(z * fst[x.(z * snd[w.(<x,w>par * Function_ f)])]):s&~t  x:s  w:~t  Function_:s&~t
	    action_coterm :: Coterm_
	    action_coterm  =
		    Abstract_ "@z" (Cut_ "action_coterm"
			(Variable_ {-z-} 0)
			(Selector_ 2 argument
			    (Abstract_ "@x" (Cut_ "action_coterm"
				(Variable_ {-z-} 1)
				(Selector_ 2 continuation (Abstract_ "@w" (Cut_ "action_coterm"
				    (Record_ [Variable_ {-x-} 1, Variable_ {-w-} 0])
				    (Function_ "primitive_dump" "dump" f)
				) unknown_stack_size
			    ))
			) unknown_stack_size))
		    ) unknown_stack_size
	    (continuation, argument) = (1, 0)

            f val = Use_ $ Builtin_ (BuiltinString $ dmp [] val)
	    dmp stk val = case val of
			Plan_ coterm -> old_show stk coterm
			Let_ e t -> dmp (map (\(n,tm,lim) -> (n, Just tm)) e) t
			otherwise -> "not an encapsulated coterm?: " ++ old_show stk val

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- Builtins

string_anyterm :: Anyterm_ t => String -> t
string_anyterm = term_to_any . Builtin_ . BuiltinString
integer_anyterm :: Anyterm_ t => Integer -> t
integer_anyterm = term_to_any . Builtin_ . BuiltinInteger
boolean_anyterm :: Anyterm_ t => Bool -> t
boolean_anyterm = term_to_any . Builtin_ . BuiltinBoolean
void_anyterm :: Anyterm_ t => t
void_anyterm = term_to_any (Record_ [])
unreached_anyterm :: Anyterm_ t => t
unreached_anyterm = coterm_to_any (Case_ [])
evaluate_anyterm :: Anyterm_ t => t
evaluate_anyterm = coterm_to_any (
	-- Rather than do this:
	--   \y<+ {y _be =:[(x)]; [(x)] _be}
	-- we use the fact that Be_ itself peforms evaluation.  It's not perfectly
	-- straightforward since Be_ returns the value but _evaluate must return not-not of that.
        --
	-- [ctx->ctx$1st[(      (Be_ [ev->ctx$0th[(ev)]:~t]):IO t
        --                Bind_ [bctx->bctx$0th[v->bctx$1st[(Be_[([(v)])])]]:t&~IO~~t]
	--                :IO~~t)]:~~t & ~IO ~~t]    
	Abstract_ "ctx" (Cut_ "evaluate_anyterm"
	    (Variable_ 0) -- ctx
	    (Selector_ 2 1 (Use_ (Bind_ "evaluate_anyterm" "left of evaluate" "right of evaluate"
		(Be_ "evaluate_anyterm"
		    (Abstract_ "ev" (Cut_ "evaluate_anyterm"
			(Variable_ 1) -- ctx
			(Selector_ 2 0 (Use_ (Variable_ 0))) -- ev
		    ) (Just 1))
		)
		(Abstract_ "bctx" (Cut_ "evaluate_anyterm"
		    (Variable_ 0 {-bctx-})
		    (Selector_ 2 0 (Abstract_ "v" (Cut_ "evaluate_anyterm"
			(Variable_ 1 {-bctx-})
			(Selector_ 2 1 (Use_
			    (Be_ "evaluate_anyterm" (Use_ (Plan_ (Use_ (Variable_ 0 {-v-})))))
			))
		    ) (Just 1)))
		) (Just 0))
	    )))
	) (Just 0)
    )

-- Conversions between by-value functions and by-name functions.
-- f is the function; a the calling context.
-- f*[not<([fst[arg]]not*a).arg>,ret.([snd[not<ret>]]not*a)]
-- f*not<<(<[arg]not>inl*a).arg,[ret.(<ret>inr*a)]not>>


useBoolean :: Term_ -> Bool
useBoolean = useBuiltin_ (\term -> case term of 
	BuiltinBoolean tf -> tf
	otherwise -> error "Boolean term required")
useInteger :: Term_ -> Integer
useInteger = useBuiltin_ (\term -> case term of
	BuiltinInteger n -> n
	otherwise -> error "integer term required")
useString :: Term_ -> String
useString = useBuiltin_ (\term -> case term of
	BuiltinString str -> str
	otherwise -> error "string term required")

useBuiltin_ :: (BuiltinValue -> t) -> Term_ -> t
useBuiltin_ extract term = case term of
	Builtin_ v -> extract v
	Let_ e t -> useBuiltin_ extract t
	otherwise -> error ("builtin term required, got " ++ old_show [] term ++ "\n")

-- Many, if not most of these builtins couldn't be implemented without the "Bound" expressions,
-- because they get at the internals of the data representation.
library_builtins :: UseIOType -> UseFunctionTypes -> MonadModel -> [(VarName, Term)]
library_builtins uio uft mm = add_no_span $ (adjust_snd uio uft mm) (library_builtins' uio)
library_builtins' uio = [
    ("fix", -- In Haskell, fix f = f (fix f) : (T->T)->T
            -- Y = \f->(\g->f(g g))(\g->f(g g))
	    TM (forallT proposition "T" (functionT (functionT (nameT "T") (nameT "T")) (nameT "T")))
	      $ GenericTypeTerm noT "S"
		(TM (functionT (functionT (nameT "S") (nameT "S")) (nameT "S"))
		  $ LambdaTerm (namePattern "f")
		    (TM noT $ ApplicationTerm
			(TM (prop $ YVT noT) $ LambdaTerm (namePattern "g")
			    (TM noT $ ApplicationTerm
				(TM noT $ ApplicationTerm
				    (TM (prop $ YVT noT) $ VariableTerm "g")
				    (TM (prop $ YVT noT) $ VariableTerm "g")
				)
				(TM noT $ VariableTerm "f")
			    )
			)
			(TM (prop $ YVT noT) $ LambdaTerm (namePattern "g")
			    (TM noT $ ApplicationTerm
				(TM noT $ ApplicationTerm
				    (TM (prop $ YVT noT) $ VariableTerm "g")
				    (TM (prop $ YVT noT) $ VariableTerm "g")
				)
			    (TM noT $ VariableTerm "f")
			    )
			)
		    )
		)
    ),
    ("iterate_", -- all I. all T. (~T->(I->I))->(I->T)
	TM (forallT proposition "I" (forallT proposition "T" (functionT (functionT (notT (nameT "T"))
		(functionT (nameT "I") (nameT "I"))) (functionT (nameT "I") (nameT "T")))))
	  $ AnnotatedTerm True (TM noT $ LambdaTerm (namePattern "iter")
	    (TM noT $ LambdaTerm (namePattern "initial")
		(TM noT
		    (let dupe =
			    TO noT $ CoabstractTerm
				(Pattern noT (AndPat
					[Just (namePattern "b"),
					 Just (namePattern "n")])
				)
				(TM noT $ VariableTerm "b")
				(TO noT $ UseTerm
				    (TM noT $ ProductTerm [
					TM noT $ VariableTerm "b",
					TM noT $ ApplicationTerm
					    (TM noT $ VariableTerm "n")
					    (TM noT $ ApplicationTerm
						(TM noT $ PlanTerm
						    (TO noT $ CovariableTerm "ret")
						)
						(TM noT $ VariableTerm "iter")
					    )
					]
				    )
				)
		     in AbstractTerm (Copattern noT (NameCopat "ret"))
			(TM noT $ ProductTerm [
			    TM noT $ PlanTerm dupe,
			    TM noT $ VariableTerm "initial"
			    ]
			)
			dupe
		    )
		)
	    )
	)
    ),
{-
    (\iter -> \initial ->
      (ret
      <+
	(&
	   (
	     [(&b&n&)
	     ->
	       b
	     $
	       [(&
		   b
		 &
		   (n ([ret]) iter)
		 &
		)
	       ]
	     ]
	   )
	 &
	   initial
	 &
	)
      $
	[(&b&n&)
	->
	  b
	$
	  [(&
	      b
	    &
	      (n ([ret]) iter)
	    &
	   )
	  ]
	]
      )
    )
-}					
    ("iterate", -- all I. all T. (I->(|I|T|)))->(I->T)
	TM (forallT proposition "I" (forallT proposition "T" (functionT
				(functionT (nameT "I") (Type.orT no_span [nameT "I", nameT "T"]))
				(functionT (nameT "I") (nameT "T")))))
	  $ AnnotatedTerm True (TM noT $ LambdaTerm (namePattern "iter")
	    (TM noT $ LambdaTerm (namePattern "initial")
		(TM noT
		    (let dupe =
			    TO noT $ CoabstractTerm
				(Pattern noT (AndPat
					[Just (namePattern "b"),
					 Just (namePattern "n")])
				)
				(TM noT $ VariableTerm "b")
				(TO noT $ UseTerm
				    (TM noT $ ProductTerm [
					TM noT $ VariableTerm "b",
					TM noT $ AbstractTerm (Copattern noT (NameCopat "continue"))
					    (TM noT $ ApplicationTerm
						(TM noT $ VariableTerm "n")
						(TM noT $ VariableTerm "iter")
					    )
					    (TO noT $ SumTerm [
						TO noT $ CovariableTerm "continue",
						TO noT $ CovariableTerm "ret"
						]
					    )
					]
				    )
				)
		     in AbstractTerm (Copattern noT (NameCopat "ret"))
			(TM noT $ ProductTerm [
			    TM noT $ PlanTerm dupe,
			    TM noT $ VariableTerm "initial"
			    ]
			)
			dupe
		    )
		)
	    )
	)
    ),
{-
    (\iter -> \initial ->
      (ret
      <+
	(&
	   (
	     [(&b&n&)
	     ->
	       b
	     $
	       [(&
		   b
		 &
		   (c <+ n iter $ [|c|ret|])
		 &
		)
	       ]
	     ]
	   )
	 &
	   initial
	 &
	)
      $
	[(&b&n&)
	->
	  b
	$
	  [(&
	      b
	    &
	      (c <+ n iter $ [|c|ret|])
	    &
	   )
	  ]
	]
      )
    )
-}					
    ("show_boolean", TM (functionT booleanT stringT) $ LambdaTerm (namePattern "tf")
			(TM stringT $ ApplicationTerm
			    (TM stringT $ StringTerm "true")
			    (TM (functionT stringT stringT) $ ApplicationTerm
				(TM stringT $ StringTerm "false")
				(TM (functionT stringT (functionT stringT stringT)) $ ApplicationTerm
				    (TM booleanT $ VariableTerm "tf")
				    (TM (functionT booleanT (functionT stringT (functionT stringT stringT))) $
					ApplicationTerm
					    (TM noT $ VariableTerm "select")
					    (TM noT $ LambdaTerm
						(specificPattern stringT (namePattern "specific_select"))
						(TM noT $ VariableTerm "specific_select")
					    )
				    )
				)
			    )
			)
    ),
    ("show_void", TM (functionT (trueT) stringT) $ LambdaTerm (namePattern "v") (TM stringT $ StringTerm "void")),
    -- error has type String -> t
    ("error", TM (forallT proposition "T" (functionT stringT (nameT "T"))) $ GenericTypeTerm noT "T"
		    (TM (functionT stringT (nameT "T")) $ LambdaTerm (namePattern "t")
			(TM noT $ AbstractTerm
			    (Copattern noT (NameCopat "ret"))
			    (TM noT $ ApplicationTerm
				(TM noT $ VariableTerm "t")
			        (TM noT $ VariableTerm "abort")
			    ) -- :False
			    (TO noT $ UnreachedTerm)
			)
		    )
    ),
    ("compose", let
		-- Ar.As.At.(r->s) -> ((s->t) -> (r->t))
		r = nameT "R"
		s = nameT "S"
		t = nameT "T"
		r2s = functionT r s
		s2t = functionT s t
		r2t = functionT r t
		f = functionT r2s (functionT s2t r2t)
		tp = forallT proposition "R" (forallT proposition "S" (forallT proposition "T" f))
		in
		    TM tp $ GenericTypeTerm noT "R"
			(TM noT $ GenericTypeTerm noT "S"
			    (TM noT $ GenericTypeTerm noT "T"
				(TM noT $ LambdaTerm (namePattern "f1")
				    (TM noT $ LambdaTerm (namePattern "f2")
					(TM noT $ LambdaTerm (namePattern "t")
					    (TM noT $ ApplicationTerm
						(TM noT $ ApplicationTerm
						    (TM noT $ VariableTerm "t")
						    (TM noT $ VariableTerm "f1")
						)
						(TM noT $ VariableTerm "f2")
					    )
					)
				    )
				)
			    )
			)
    ),
    ("bind", let
		--  (s -> iOVT t) -> iOVT s -> iOVT t
		s = nameT "A"
		t = nameT "B"
		ios = iOVT s
		iot = iOVT t
		fsiot = functionT s iot
		f = functionT fsiot (functionT ios iot)
		tp = forallT proposition "A" (forallT proposition "B" f)
		in
		    TM tp $ GenericTypeTerm noT "A"
			(TM noT $ GenericTypeTerm noT "B"
			    (TM noT $ LambdaTerm (namePattern "binder")
				(TM noT $ LambdaTerm (namePattern "act")
				    (TM noT $ BindTerm
					(TM noT $ VariableTerm "act")
					(TM noT $ VariableTerm "binder")
				    )
				)
			    )
			)
    ),
    ("trace", -- all B.B->IOV B
              -- (@B:> \v -> {(("--- observed: " (v dump (\@d->d)) concat) " ---" concat) put; v be (\@b->b)})
		TM (forallT proposition "B" (functionT (nameT "B") (iOVT (nameT "B")))) $ GenericTypeTerm noT "B"
		    (TM noT $ LambdaTerm (namePattern "v")
			(TM noT $ SeqTerm
			    (TM noT $ ApplicationTerm -- of put
				(TM noT $ ApplicationTerm -- of concat
				    (TM noT $ ApplicationTerm -- of concat
					(TM stringT $ StringTerm "--- observed: ")
					(TM noT $ ApplicationTerm -- partial of concat
					    (TM noT $ ApplicationTerm -- of \@d->d
						(TM {-(nameT "T")-} noT $ VariableTerm "v")
						(TM noT $ ApplicationTerm
						    (TM noT $ VariableTerm "dump")
						    (TM noT $ LambdaTerm
							(specificPattern noT (namePattern "specific_dump"))
							(TM noT $ VariableTerm "specific_dump")
						    )
						)
					    )
					    (TM noT $ VariableTerm "concat")
					)
				    )
				    (TM noT $ ApplicationTerm -- partial of concat
					(TM stringT $ StringTerm " ---")
					(TM noT $ VariableTerm "concat")
				    )
				)
				(TM noT $ VariableTerm "put")
			    )
			    (TM noT $ ApplicationTerm
				(TM noT $ VariableTerm "v")
				(TM noT $ ApplicationTerm
				    (TM noT $ VariableTerm "be")
				    (TM noT $ LambdaTerm
					(specificPattern noT (namePattern "specific_be"))
					(TM noT $ VariableTerm "specific_be")
				    )
				)
			    )
			)
		    )
    )
    ]

library_cobuiltins :: UseIOType -> UseFunctionTypes -> MonadModel -> [(VarName, Coterm)]
library_cobuiltins uio uft mm = add_no_span $ (adjust_snd uio uft mm)
	$ ((filter is_good_builtin $ map coize' (library_builtins' uio)) ++ library_cobuiltins'')
  where
    UseStrictMonad sm = mm
    UseLowerIOType lio = uio
    library_cobuiltins'' = library_cobuiltins' uio
    native_cobuiltin_names = map fst library_cobuiltins''
    is_good_builtin (nm,_) = case nm of
		_ | nm `elem` native_cobuiltin_names -> False
		otherwise -> True
    coize' (nm, tm) = ("_" ++ nm, coize tm)
    coize tm@(TM tp (StringTerm str)) = TO (notT stringT) (UseTerm tm)
    coize (TM tp t) = TO (czt tp) (cz t)
    is_strict_type tpnm = case tpnm of
	    "False" -> True
	    "True" -> True
	    "String" -> True
	    "Boolean" -> True
	    -- The type parameters of some generic functions can be converted to lazy types,
	    -- and others must remain strict, translating A to ~A.
	    (x:xs) | sm && 'A' <= x && x <= 'C' -> True
	    otherwise -> False
    czt typ@(Type kind sp tp) = Type kind sp $ case tp of
	NoT -> NoT
	NameT tpnm _ | is_strict_type tpnm -> ApplicationT (ctorT_ sp NotT) typ
	NameT nm x -> NameT nm x
	FunctionT from_t to_t -> CofunctionT (czt from_t) (czt to_t)
	ApplicationT (Type k' sp' (NameT "IOV" x)) tp' -> if not sm
		then ApplicationT (Type k' sp' (NameT "ION" x)) (czt tp')
		else if lio then ApplicationT (ctorT_ sp NotT) (Type Proposition sp (ApplicationT (Type k' sp' (NameT "IO" "S")) tp'))
		else error "strict monads but can't lower IO"
	ApplicationT (Type k' sp' NotT) tp' -> ApplicationT (Type k' sp' NotT) (czt tp')
	YVT tp -> YNT (czt tp)
	ExistsT k nm tp' -> ForallT k nm (czt tp')
	ForallT k nm tp' -> ExistsT k nm (czt tp')
	OrT tps -> AndT PlainOrigin (map czt tps)
	AndT origin tps -> OrT (map czt tps)
	
	otherwise -> error $ "library_cobuiltins czt:@" ++ show tp
    cztpex (TypeExpression tp) = TypeExpression (czt tp)
    cz term = case term of
	LambdaTerm pat tm -> ColambdaTerm (copatize pat) (coize tm)
	ApplicationTerm arg func -> CoapplicationTerm (coize arg) (coize func)
	VariableTerm nm -> CovariableTerm ('_' : nm)
	BindTerm act followon -> CobindTerm (coize act) (coize followon)
	SeqTerm act1 act2 -> CoseqTerm (coize act1) (coize act2)
	VoidTerm -> UnreachedTerm
	ProductTerm tms -> SumTerm (map coize tms)
	AbstractTerm copat t c -> CoabstractTerm (cocopatize copat) (cocoize c) (coize t)
	GenericTypeTerm k tp tm -> AbstractTypeTerm k tp (coize tm)
	AnnotatedTerm tf t -> CoannotatedTerm tf (coize t)
	PlanTerm c -> UseTerm (cocoize c)
	otherwise -> error $ "library_cobuiltins cz of " ++ tshow TopShowSpan False False term
    cocoize' (nm, ctm) = (nm, cocoize ctm)
    cocoize ctm@(TO tp UnreachedTerm) = TM (notT falseT) (PlanTerm ctm)
    cocoize (TO tp c) = TM (czt tp) (cocz c)
    cocz term = case term of
	CoabstractTerm pat t c -> AbstractTerm (copatize pat) (cocoize c) (coize t)
	ColambdaTerm copat c -> LambdaTerm (cocopatize copat) (cocoize c)
	AbstractTypeTerm k tp ctm -> GenericTypeTerm k tp (cocoize ctm)
	CovariableTerm nm -> VariableTerm (case nm of {('_':nm') -> nm'; otherwise-> nm})
	UseTerm t -> PlanTerm (coize t)
	SumTerm tms -> ProductTerm (map cocoize tms)
	otherwise -> error $ "library_cobuiltins cocz of " ++ tshow TopShowSpan False False term
    copatize (Pattern ptp pat) = Copattern (czt ptp) $ case pat of 
	NamePat nm -> NameCopat $ "_" ++ nm
	VoidPat -> UnreachedCopat
	AndPat mpats -> OrCopat (map (fmap copatize) mpats)
	NotPat copat -> NotCopat $ cocopatize copat
	SpecificPat tp pat -> PackageCopat (cztpex tp) $ copatize pat
    cocopatize (Copattern ptp copat) = case copat of
	NameCopat nm ->
		case nm of
			('_':nm') -> namePattern nm'
			otherwise -> namePattern nm
	UnreachedCopat -> Pattern noT $ VoidPat
	OrCopat mpats -> Pattern noT $ AndPat (map (fmap cocopatize) mpats)
	NotCopat pat -> Pattern noT $ NotPat $ copatize pat
	PackageCopat tp copat -> specificPattern noT $ cocopatize copat
library_cobuiltins' uio = [
    -- [@T<: \t<+[no_return -> ()${[[("error: ")] t _concat] _put;_put_newline} _abort [\@pa<+pa]]]
    ("_error", TO (existsT proposition "T" (cofunctionT (notT stringT) (nameT "T"))) $ AbstractTypeTerm noT "T"
		    (TO (cofunctionT (notT stringT) (nameT "T")) $ ColambdaTerm (nameCopattern "t")
			(TO noT $ CoabstractTerm
			    (Pattern noT (NamePat "ret"))
			    (TM noT $ VoidTerm)
			    (TO noT $ CoapplicationTerm
				(TO noT $ CovariableTerm "t")
				(TO noT $ CovariableTerm "_abort")
			    )
			)
		    )
    )
    ]

add_no_span :: (Functor t) => [(String, t Type)] -> [(String, t (Type, Span))]
add_no_span = map (\(nm, binding) -> (nm, fmap (\tp -> (tp,no_span)) binding))

adjust_snd :: (Functor t) => UseIOType -> UseFunctionTypes -> MonadModel -> [(String, t Type)] -> [(String, t Type)]
adjust_snd uio ufn mm = map (\(nm, binding) -> (nm, fmap (lower_type uio ufn mm) binding))
lower_type :: UseIOType -> UseFunctionTypes -> MonadModel -> Type -> Type
lower_type (UseLowerIOType lio) (UseLowerFunctionTypes lfn) mm = (if lio then lower_io_type mm else id) . (if lfn then lower_fn_type else id)
lower_io_type :: MonadModel -> Type -> Type
lower_io_type mm@(UseStrictMonad strict_monad) (Type k sp tp) = Type k sp $ case tp of
	ApplicationT (Type k sp (NameT "IOV" _)) tp' -> ApplicationT (ctorT_ sp (NameT "IO" "V")) (lower_io_type mm tp')
	ApplicationT (Type k sp (NameT "ION" _)) ty'@(Type _ sp' tp') -> case tp' of
		ApplicationT (Type _ _ NotT) tp'' | strict_monad
			-> ApplicationT (ctorT_ sp NotT) (Type Proposition sp (ApplicationT (Type k sp (NameT "IO" "S")) (lower_io_type mm tp'')))
		_ | not strict_monad
			-> ApplicationT (ctorT_ sp NotT) (Type Proposition sp (ApplicationT (Type k sp (NameT "IO" "N"))
										(Type.notT sp (lower_io_type mm ty'))))
		_ -> error "lowering unexpected ION type"
	otherwise -> tp_map tp
    where
	tp_map = fmap (lower_io_type mm) -- for Tp's
lower_fn_type :: Type -> Type
lower_fn_type (Type k sp tp) = Type k sp $ case tp of
	FunctionT atp btp@(Type bk bsp _) -> lower_function_type sp bsp (lower_fn_type atp) (lower_fn_type btp)
	CofunctionT atp@(Type ak asp _) btp -> lower_cofunction_type asp (lower_fn_type atp) (lower_fn_type btp)
	otherwise -> tp_map tp
    where
	tp_map = fmap lower_fn_type -- for Tp's

complement_bif_type :: UseIOType -> MonadModel -> Type -> Type
complement_bif_type uio@(UseLowerIOType lio) mm@(UseStrictMonad sm) typ@(Type k sp tp) = Type k sp $
        case tp of
	    FunctionT from_t to_t -> CofunctionT (lt from_t) (lt to_t)
	    ApplicationT (Type k' sp' (NameT "IOV" x)) arg | sm && lio
		-- This is sketchy.  It certainly doesn't work in general.
		-> ApplicationT (ctorT_ sp' NotT) (Type k sp (ApplicationT (Type k' sp' (NameT "IO" "S")) arg))
	    ApplicationT constr arg -> ApplicationT (lt constr) (lt arg)
	    LambdaT k nm t -> LambdaT k nm (lt t)
	    NameT "IOV" x -> if not sm then NameT "ION" x else error "strict monads with bif"
	    NameT "ION" x -> NameT "IOV" x
	    NameT "String" _ -> ApplicationT (ctorT_ sp NotT) typ
	    NameT "Integer" _ -> ApplicationT (ctorT_ sp NotT) typ
	    NameT "Boolean" _-> ApplicationT (ctorT_ sp NotT) typ
	    NameT "True" _ -> ApplicationT (ctorT_ sp NotT) typ
	    NameT "False"_ -> ApplicationT (ctorT_ sp NotT) typ
	    AndT _ ts -> OrT $ map lt ts
	    OrT ts -> AndT PlainOrigin $ map lt ts
	    NotT -> NotT
	    NameT "List" x -> NameT "Colist" x
	    NameT "Colist" x -> NameT "List" x
	    YVT tp -> YNT (lt tp)
	    YNT tp -> YVT (lt tp)
	    NameT nm x -> NameT nm x
	    
	    ForallT k nm tp -> ExistsT k nm $ lt tp
	    ExistsT k nm tp -> ForallT k nm $ lt tp
	    NoT -> NoT
	    otherwise -> error $ "complement_bif_type " ++ show tp
    where
	lt = complement_bif_type uio mm

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- Evaluation & Values

primitive_exit :: Anyterm_ t => t  -- IO False
primitive_exit = term_to_any (Exit_ "primitive_exit")
primitive_abort :: Anyterm_ t => String -> t  -- String -> False
primitive_abort msg = oblivious_anyterm (Cut_ "primitive_abort"
					      (Error_ msg)
					      (Case_ [])
					)

-- Typically, one would give this term the type "FalseT", but any type will do.
-- This is an appropriate behavior for "bottom".
hang_anyterm :: Anyterm_ t => t
hang_anyterm = term_to_any $ Error_ "If this were a complete implementation, a thread would be hanging."

primitive_ignore :: Term_ -> Term_  -- t -> True
primitive_ignore val = Record_ []

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- If one name relies on another, it must not precede.
--
native_builtins :: Anyterm_ trm => UseIOType -> UseFunctionTypes -> MonadModel -> Either Handle (IORef (Maybe Handle)) -> ([(VarName, trm)], [(VarName, (Way, Type))])
native_builtins uio ufn monad_model runtime_output = fixup $ map (\(nm,tp,val) -> (bif_name_of nm, bif_type_of nm tp, val)) $ filter filter_way bis
 where
  bis = [
    ("head", forallT proposition "T" (functionT (listT (nameT "T")) (nameT "T")),
		lambda_anyterm (\t -> case useList t of
						(h:hs) -> h
					        [] -> error "head of null list") "head"),
    ("tail", forallT proposition "T" (functionT (listT (nameT "T")) (listT (nameT "T"))),
		lambda_anyterm (\t -> case useList t of
						(h:hs) -> list_anyterm hs
						[] -> error "tail of null list") "tail"),
    ("cons", forallT proposition "T" (functionT (nameT "T") (functionT (listT (nameT "T")) (listT (nameT "T")))),
		lambda_anyterm (\t -> lambda_anyterm ((\l -> list_anyterm (term_to_any t:l)) . useList) "cons") "cons"),
    ("plus", functionT integerT (functionT integerT integerT),
		lambda_anyterm ((\n -> lambda_anyterm ((\n' -> integer_anyterm (n + n')) . useInteger) "plus") . useInteger) "plus"),
    ("minus", functionT integerT (functionT integerT integerT),
		lambda_anyterm ((\n -> lambda_anyterm ((\n' -> integer_anyterm (n' - n)) . useInteger) "minus") . useInteger) "minus"),
    ("times", functionT integerT (functionT integerT integerT),
		lambda_anyterm ((\n -> lambda_anyterm ((\n' -> integer_anyterm (n * n')) . useInteger) "times") . useInteger) "times"),
    ("concat", functionT stringT (functionT stringT stringT),
		lambda_anyterm ((\str -> lambda_anyterm ((\str' -> string_anyterm (str' ++ str)) . useString) "concat") . useString) "concat"),
    ("substring", functionT integerT (functionT integerT (functionT stringT stringT)),
		lambda_anyterm ((\end ->
		     lambda_anyterm ((\begin ->
			   lambda_anyterm ((\str ->
				                  if 0 <= begin && begin <= end && end <= toInteger (length str)
						  then string_anyterm (take (fromInteger (end - begin)) (drop (fromInteger begin) str))
						  else error ("(" ++ show str ++ " " ++ show begin ++ " " ++ show end ++ " substring)")) . useString)
							"substring3") . useInteger) "substring2") . useInteger) "substring1"),
    ("string_length", functionT stringT integerT,
		lambda_anyterm ((\str -> integer_anyterm (toInteger (length str))) . useString) "string_length"),
    ("put", functionT stringT (iOVT no_value_type),
		lambda_anyterm (\term -> io_anyterm monad_model "put" no_value_type (do
					(Just ro) <- case runtime_output of
						Left x -> return (Just x)
						Right ref -> readIORef ref
					hPutStr ro (useString term)
					return no_value_term
			)) "put"),
    ("put_newline", iOVT no_value_type, io_anyterm monad_model "put_newline" no_value_type (do
					(Just ro) <- case runtime_output of
						Left x -> return (Just x)
						Right ref -> readIORef ref
					hPutStrLn ro ""
					return no_value_term
			)),
    ("be", forallT proposition "T" (functionT (nameT "T") (iOVT (nameT "T"))),
		function_anyterm (be monad_model) "be" empty_stack),
    ("evaluate", forallT proposition "T" (functionT (notT (nameT "T")) (iOVT (notT (nameT "T")))), evaluate_anyterm),
    ("equals", functionT integerT (functionT integerT booleanT),
		lambda_anyterm ((\n -> lambda_anyterm ((\n' -> boolean_anyterm (n == n')) . useInteger) "equals") . useInteger) "equals"),
    ("and", functionT booleanT (functionT booleanT booleanT),
		lambda_anyterm ((\tf -> lambda_anyterm ((\tf' -> boolean_anyterm (tf && tf')) . useBoolean) "and") . useBoolean) "and"),
    ("or", functionT booleanT (functionT booleanT booleanT),
		lambda_anyterm ((\tf -> lambda_anyterm ((\tf' -> boolean_anyterm (tf || tf')) . useBoolean) "or") . useBoolean) "or"),
    ("not", functionT booleanT booleanT, lambda_anyterm (boolean_anyterm . not . useBoolean) "not"),
    ("false", booleanT, boolean_anyterm False),
    ("true", booleanT, boolean_anyterm True),
    ("void", trueT, void_anyterm),
    ("hang", falseT, hang_anyterm),
    ("unreached", falseT, unreached_anyterm),
    ("select", forallT proposition "T" (functionT booleanT (functionT (nameT "T") (functionT (nameT "T") (nameT "T")))),
	    lambda_anyterm ((\tf -> case tf of
		False -> function_anyterm (\v' -> function_anyterm (\v'' -> v') "select False") "select False" empty_stack
		True -> function_anyterm (\v' -> function_anyterm (\v'' -> v'') "select True") "select True" empty_stack) . useBoolean) "select"),
    ("show_integer", functionT integerT stringT, lambda_anyterm (string_anyterm . show . useInteger) "show_integer"),
    ("show_string", functionT stringT stringT, lambda_anyterm (string_anyterm . show . useString) "show_string"),
    -- ("show_list", -- takes a list and a "show" function
    --    actionTerm (\show -> actionTerm ((show_list show) . useList) "show_list") "show_list"),

    -- exit does not display anything. IO False
    ("exit", iOVT falseT, primitive_exit),
    -- abort presents a diagnostic when exiting. Type String -> False
    ("abort", functionT stringT falseT,
		lambda_anyterm (primitive_abort . useString) "abort"),
    -- Dumps an argument of any type. Dumps its value in by-value; dumps its guts in by-name.
    ("dump", forallT proposition "T" (functionT (nameT "T") stringT),
		primitive_dump), -- type t -> String
    ("ignore", forallT proposition "T" (functionT (nameT "T") trueT),
		lambda_anyterm (term_to_any . primitive_ignore) "ignore") -- type t -> True
    ]
    where
{-
	-- show_list :: Term_ -> [Term_] -> Term_
	show_list show_element xs = case xs of
		[] -> string_anyterm "[]"
		otherwise -> let
			show m = apply stringT m show_element
			-- strs :: [Term_]
			strs = map show xs
			guts = foldr1 (\x -> \y -> x ++ "," ++ y)
			-- subbracketer :: [Term_] -> [String] -> Term_
			subbracketer [x] = \ls -> lambda (\m_str -> string_anyterm ("[" ++ guts (m_str:ls) ++ "]"))
			subbracketer (x:xs) = \ls -> lambda (\n_str -> subbracketer xs (n_str:ls))
			subbracketer [] = undefined -- can't get here
			bracketer = subbracketer strs []
			in foldr (apply noT) bracketer strs
	    where
		apply tp m e = byValueApplication "application show_term" m e
		lambda f = lambda_anyterm (term_to_any . f . useString) "show_list"
-}
	no_value_term = case way of
		ByValue -> Record_ []
		ByName -> Record_ []
		ByType -> undefined
	no_value_type = case way of
		ByValue -> trueT
		ByName -> trueT
		ByType -> undefined
	expr_name = case way of
		ByValue -> "term"
		ByName -> "coterm"
		ByType -> "type"
	empty_stack = create_stack empty

  way = let (x,y,z) = head bis in which_way z
  bif_type_of nm tp = (lower_type uio ufn monad_model) $ case way of
	ByValue -> tp
	ByName -> case nm of
		"unreached" -> falseT
		"abort" -> cofunctionT (notT stringT) trueT
		_ | not lio -> complement_bif_type uio monad_model tp
		_ | not sm -> complement_bif_type uio monad_model tp
		"be" -> existsT proposition "T" (cofunctionT (notT (nameT "T")) (notT (iOT_N monad_model (nameT "T"))))
		otherwise -> complement_bif_type uio monad_model tp
	ByType -> undefined
     where
	UseStrictMonad sm = monad_model
	UseLowerIOType lio = uio
  bif_name_of nm = case way of
	ByName -> '_':nm
	ByValue -> nm
	ByType -> '-':nm
  filter_way :: Anyterm_ trm => (VarName, Type, trm) -> Bool
  filter_way (nm, _, _) = case nm of
	"unreached" -> way == ByName
	"evaluate" -> case monad_model of
		UseStrictMonad False -> way == ByName
		otherwise -> False  -- _be is a better name.
	otherwise -> True

fixup :: Anyterm_ trm => [(VarName, Type, trm)] -> ([(VarName, trm)], [(VarName, (Way, Type))])
fixup vars = (vars', f_map)
  where
    vars' = map (\(v,tp,ftrm) -> (v,ftrm)) vars
    f_map = map (\(v,tp,trm) -> (v,(which_way trm, tp))) vars

-- List has the more primitive names, especially types, first.
types_of_native_builtins :: UseIOType -> UseFunctionTypes -> MonadModel
			-> [(VarName, (Way, Type))]
types_of_native_builtins uio@(UseLowerIOType lio) ufn mm
	= types_of_types ++ types_of_builtins ++ types_of_builtins'
    where
	(nb::[(VarName, Term_)], types_of_builtins::[(VarName, (Way, Type))])
		= native_builtins uio ufn mm (Left stdout)
	(nb'::[(VarName, Coterm_)], types_of_builtins'::[(VarName, (Way, Type))])
		= native_builtins uio ufn mm (Left stdout)
	types_of_types = map kind_to_type kinds_of_atomic_types
		++ if lio then io_type_names mm else io_atomic_types
kind_to_type (name, Kind kind) = (name, (ByType, Type kind no_span NoT))

kinds_of_atomic_types = [
	("True", kprop),
	("False", kprop),
	("String", kprop),
	("Integer", kprop),
	("Boolean", kprop),
	("List", Kind $ Predicate kprop kprop),
	("Colist", Kind $ Predicate kprop kprop),
	("IO", Kind $ Predicate kprop kprop)
	]
io_atomic_types = map kind_to_type [
	("IOV", Kind $ Predicate kprop kprop),
	("ION", Kind $ Predicate kprop kprop)
	]
io_type_names mm@(UseStrictMonad sm) = [
	("IOV", (ByType, Type (Predicate kprop kprop) no_span
	                 (LambdaT (Type Proposition no_span NoT) "T" (iOT_V (nameT "T")))
	))] ++ if sm then [] else [
	("ION", (ByType, Type (Predicate kprop kprop) no_span
	                 (LambdaT (Type Proposition no_span NoT) "T" (notT (iOT_N mm (notT (nameT "T")))))
	))]

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
applyMany :: [(a -> a)] -> a -> a
applyMany [] x = x
applyMany (f:fs) x = applyMany fs (f x)

substituteLibraryBuiltins :: Expression e => UseIOType -> UseFunctionTypes -> MonadModel -> e -> e
substituteLibraryBuiltins uio ufn mm stmt = (coresult . result) stmt
  where
    (coenames, enames) = partition is_coname $ namesOf stmt
    bnames = map fst builtins
    snames = filter (const True . (`elem` enames)) bnames
    cobnames = map fst cobuiltins
    cosnames = filter (const True . (`elem` coenames)) cobnames
    result = applyMany (map substituteBuiltin snames)
    -- substituteBuiltin :: String -> e -> e
    substituteBuiltin name = transform
      where
	transform = maybe id
			  (\val -> substitute name (Left val::Either Term Coterm))
			  (lookup name builtins)
    coresult = applyMany (map substituteCobuiltin cosnames)
    -- substituteCobuiltin :: String -> e -> e
    substituteCobuiltin name = transform
      where
	transform = maybe id
			  (\val -> substitute name (Right val::Either Term Coterm))
			  (lookup name cobuiltins)
    is_coname nm = (take 1 nm) == "_"
    builtins = library_builtins uio ufn mm
    cobuiltins = library_cobuiltins uio ufn mm

nativeBuiltins :: UseIOType -> UseFunctionTypes -> MonadModel -> IORef (Maybe Handle) -> Map String (Either Term_ Coterm_)
nativeBuiltins uio ufn mm runtime_output = fromList $ map convert_builtin_term nb
				++ map convert_builtin_coterm nb'
    where
	(nb::[(VarName, Term_)], _) = native_builtins uio ufn mm (Right runtime_output)
	(nb'::[(VarName, Coterm_)], _) = native_builtins uio ufn mm (Right runtime_output)
	convert_builtin_term (nm, t) = (nm, Left t)
	convert_builtin_coterm (nm, c) = (nm, Right c)
	
