{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Evaluate (evaluate', base_expose_id) where

import Control.Monad

import Basic
import Intermediate
import Parameters
import Builtin
import Strict
import Type
import IO (hPutStrLn, hPutStr, stderr)

base_expose_id = 1

isValue :: Term_ -> Bool
isValue t = case t of
	Record_ ts -> all isValue ts
	Enumerator_ m n t -> isValue t
	Plan_ t -> True
	Builtin_ (BuiltinList ts) -> all isValue ts
	Builtin_ _ -> True
	Exit_ _ -> True
	Do_ _ _ _ -> True
	Be_ _ c -> True
	Bind_ _ _ _ deliverer receiver -> isValue deliverer
	Coabstract_ _ _ _ -> False
	Error_ _ -> False
	Variable_ _ -> True
	Let_ e t -> isValue t

substitute_var :: [(VarName, Either Term_ Coterm_, Maybe Int)] -> Term_ -> (Bool, Term_)
substitute_var stack term = case term of
	Record_ ts -> (or subs, Record_ tsu)
	    where
		(subs, tsu) = unzip (map (substitute_var stack) ts)
	Enumerator_ m n t -> (sub, Enumerator_ m n tu)
	    where
		(sub, tu) = substitute_var stack t
	Plan_ t -> (False, term)
	Builtin_ (BuiltinList ts) -> (or subs, Builtin_ (BuiltinList tsu))
	    where
		(subs, tsu) = unzip (map (substitute_var stack) ts)
	Builtin_ _ -> (False, term)
	Exit_ _ -> (False, term)
	Do_ _ _ _ -> (False, term)
	Be_ _ c -> (False, term)
	Bind_ orig opd1str opd2str deliverer receiver -> (sub, Bind_ orig opd1str opd2str d receiver)
	    where
		(sub, d) = substitute_var stack deliverer
	Coabstract_ _ _ _ -> (False, term)
	Error_ _ -> (False, term)
	Variable_ offset -> case stack !! offset of
		(nm, Left t, lim) -> (True, t)
		(nm, Right c, lim) -> error $ "Variable refers to covalue " ++ nm ++ "."
	Let_ e t -> (sub, Let_ e tu)
	    where
		(sub, tu) = substitute_var e t

-- IsCovalue is not needed since evaluate is by-value in bias.

evaluate' params maybe_handle = evaluate 0 [] [] base_expose_id
  where
    -- tracing
    tr = trace_evaluation params
    qs = show_quick_stack params
    ls = show_long_stack params
    ce = compressed_evaluation params

    verbose = if tr then maybe_handle else Nothing

    restructure_term :: String -> Int -> [(VarName, Either Term_ Coterm_, Maybe Int)] -> [(VarName, Either Term_ Coterm_, Maybe Int)] ->
			Integer -> Coterm_ -> (Term_, Term_ -> Term_) -> IO ()
    restructure_term orig compress_count stack costack next_expose coop (op, context) = evaluate compress_count stack' stack' next_expose
	    (Cut_ orig
		  (Let_ stack op)
		  (Abstract_ "@z" (Cut_ orig
					(context (Variable_ 0))
					(Covariable_ 1)) (Just 2)))
	where
	    stack' = limit_stack "stack" (Just 1) $ ("@a", Right $ Colet_ costack coop, Just $ length stack + 1) : stack

    {-
    Evaluation yields substantial results as follows:
       A coterm of "action" or IOU type (e.g. a bind operator) is cut with a run_term.
       Evaluating the expression runs the actions as a side-effect.
    -}

    evaluate :: Int -> [(VarName, Either Term_ Coterm_, Maybe Int)] -> [(VarName, Either Term_ Coterm_, Maybe Int)] -> Integer -> Statement_ -> IO ()
    evaluate = evaluate'' True
    evaluate'' real compress_count stack costack next_expose x@(Cut_ orig tt tc) = do
	    let (modded, tt') = substitute_var stack tt
	    case verbose of
		Just h | real -> do
		    hPutStrLn h ("    " ++ "stack:")
		    when (qs || not ls) $ hPutStr h $ stack_show True "    " stack
		    when ls $ hPutStr h $ stack_show False "    " stack
		    hPutStrLn h ("    " ++ "costack:")
		    when (qs || not ls) $ hPutStr h $ stack_show True "    " costack
		    when ls $ hPutStr h $ stack_show False "    " costack
		    hPutStrLn h ("ev: " ++ qshow tt ++ " $ " ++ qshow tc)
		    if modded then do
			    hPutStrLn h " sub"
			    hPutStrLn h ("    " ++ qshow tt' ++ " $ " ++ qshow tc)
			else return ()
		    hPutStrLn h ("oldev: " ++ old_show (map (\(n,tm,lim) -> (n, Just tm)) stack) tt'
					   ++ " $ "
					   ++ old_show (map (\(n,tm,lim) -> (n, Just tm)) costack) tc)
		otherwise -> return ()
	    case tt' of
		Record_ ts -> maybe (pass tt') restructure (find_list_restructure Record_ id ts)
		Enumerator_ m n t -> if isValue t then (pass tt')
					else restructure (t, \t' -> Enumerator_ m n t')
		Plan_ t -> pass tt'
		Builtin_ (BuiltinList ts) -> maybe (pass tt') restructure
						   (find_list_restructure list_anyterm id ts)
		Builtin_ _ -> pass tt'
		Exit_ _ -> pass tt'
		Do_ _ _ _ -> pass tt'
		Be_ _ c -> pass tt'
		Bind_ orig opd1str opd2str deliverer receiver -> if isValue deliverer then pass tt'
					else restructure (deliverer, \d -> Bind_ (orig++"'") opd1str opd2str d (Colet_ stack receiver))
		Coabstract_ nm stmt stack_size -> let
		    -- next_stmt = substitute_coterm name tc stmt
		    stack' = limit_stack "stack" stack_size
				$ (nm, Right $ Colet_ costack tc, Just $ length stack + 1) : stack
		    -- in evaluate stack' stack' stmt
		    in if compress_count <= 0
			then do
			    let (sz, (stack'', stmt')) = compress ce
						(map (\(n,tm,lim) -> (n, Just tm)) stack')
						(stack', stmt)
			    aside verbose $ "size co: " ++ show sz
			    seq (strict stack'') $ seq (strict stmt')
						 $ evaluate (truncate sz) stack'' stack'' next_expose stmt'
			else evaluate (compress_count - 1) stack' stack' next_expose stmt
		Error_ msg -> die ("--- error ---" ++ case msg of {"" -> ""; _ -> "\n" ++ msg})
		Let_ stack' t -> evaluate'' False count_down stack' costack next_expose $ Cut_ orig t tc
		otherwise -> die ("evaluate by value CutStatement_: " ++ show tt')
	where
	    restructure = restructure_term orig count_down stack costack next_expose tc
	    find_list_restructure :: ([Term_] -> Term_) -> ([Term_] -> [Term_]) -> [Term_]
						        -> Maybe (Term_, Term_ -> Term_)
	    find_list_restructure comp pre [] = Nothing
	    find_list_restructure comp pre (t:ts)
					    | isValue t = find_list_restructure comp (pre.(b t:)) ts
					    | otherwise = Just (t, \t -> comp (pre (t:map b ts)))
		where
		    b t = Let_ stack t
	    pass tt = case (tt, tc) of
		    (_, Covariable_ offset)
			    -> case costack !! offset of
				    (nm, Left t, lim) -> die $ "Covariable refers to value " ++ nm ++ "."
				    (nm, Right c, lim) -> evaluate count_down stack costack next_expose $ Cut_ orig tt c
		    (_, Colet_ costack' c)
			    -> evaluate'' False count_down stack costack' next_expose $ Cut_ orig tt c
		    (Record_ ts, Selector_ m n c) | length ts == m
			    -> evaluate count_down stack costack next_expose $ Cut_ orig (ts !! n) c
		    (Enumerator_ m n t, Case_ ts) | length ts == m
			    -> evaluate count_down stack costack next_expose $ Cut_ orig t (ts !! n)
		    (Plan_ c, Use_ t)
			    -> evaluate count_down costack stack next_expose $ Cut_ orig t c
		    (_, Abstract_ nm stmt stack_size) -> let
			-- !!! (substitute_term name tt stmt)
			costack' = limit_stack "costack" stack_size
			    $ (nm, Left $ Let_ stack tt, Just $ length costack + 1) : costack
			-- in evaluate costack' costack' stmt
			in if compress_count <= 0
			    then do
				let (sz, (costack'', stmt'))
					= compress ce (map (\(n,tm,lim) -> (n, Just tm)) costack')
						      (costack', stmt)
				aside verbose $ "size:    " ++ show sz
				seq (strict costack'') $ seq (strict stmt')
						     $ evaluate (truncate sz) costack'' costack'' next_expose stmt'
			    else evaluate (compress_count - 1) costack' costack' next_expose stmt
		    (Record_ [arg, cont], Function_ f_orig name f)
			-> evaluate count_down stack stack next_expose $ Cut_ orig cont (map_ (update_origin f_orig) (f arg))
		    (_, Expose_ _ (ident, orig') _) | next_expose /= ident
			-> die $ maybe "internal error: Expose got an odd value."
				    (\origin -> "--- error ---"
					++ "\nTwo actions for" ++ frame orig'
					++ "have been evaluated."
					++ "\nThe extra action, beginning with" ++ frame (origin ++ ",") ++ "is blocked.")
				    (m_origin tt)
			where
			   m_origin tt = case tt of
				Exit_ orig -> Just $ orig
				Do_ orig _ _ -> Just $ orig
				Be_ orig _ -> Just $ orig
				Bind_ orig _ _ tt' _ -> Just $
					maybe
					    (orig ++ " [but couldn't dig deeper into its left operand]")
					    id
					    (m_origin tt')
				Let_ stk tt' -> m_origin tt'
				otherwise -> Nothing
			   frame str = if '\n' `elem` str
				then concatMap ("\n    " ++) (lines str) ++ "\n"
				else " " ++ str ++ " "
		    (Exit_ _, Expose_ unreached _ _) -> return ()
		    (Do_ _ name iot, Expose_ c id (Just next)) -> do
				next_term <- iot
				unless (isValue next_term) (die "internal error: IO action returned a non-value.")
				evaluate count_down [] costack next $ Cut_ orig next_term c
		    (Be_ _ c_to_evaluate, Expose_ c id (Just next)) -> evaluate count_down costack stack next $ Cut_ orig (Plan_ c) c_to_evaluate
		    (Bind_ _ opd1str opd2str deliverer receiver, this@(Expose_ c (id, orig) mnext))
			-- 1) Form an Expose_ coterm of type IO s whose computation is
			--	    o Take the s value and combine it with a continuation
			--	      of the current Expose_ coterm to get a new value.
			--	    o Feed that value to the bind's receiver, and evaluate.
			--	i.e. [x->(&x&([this])&)$receiver]
			-- 2) Combine the deliverer with the new coterm.
			--      i.e. deliverer $ exp[x.(<x,[this]pln>rec$receiver)]
			-> evaluate count_down stack [] (fst first_offspring_expose)
				$ Cut_ orig deliverer (Expose_ (Abstract_ "@expose" (Cut_ orig (Record_ [Variable_ 0, Plan_ (Colet_ costack (Expose_ c second_offspring_expose mnext))]) (Colet_ stack receiver)) (Just 1)) first_offspring_expose (Just (fst second_offspring_expose)))
		      where
			first_offspring_expose = (2*id, opd1str)
			second_offspring_expose = (2*id+1, opd2str)
		    otherwise
			    -> mismatch tt
	    mismatch tt = die ("mismatch in cut: " ++ qshow tt ++ " $ " ++ (case tc of
			Expose_ _ id next -> qshow tc ++ " id " ++ show id ++ " next " ++ show next
			_ -> qshow tc))

	    count_down = case compress_count of
				0 -> 0
				otherwise -> compress_count - 1
	    truncate sz = if sz > fromIntegral (maxBound::Int) then maxBound else fromIntegral sz

limit_stack wher mlim stk@((head1,head2,len):tail) = case mlim of
			Nothing -> stk
			Just lim -> (head1,head2,Just lim):tail
    where
	msg = "pushing " ++ wher ++ ", new size " ++ show (length stk) ++ " expected size " ++ show mlim
limit_stack wher mlim [] = []
	
