{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Intermediate (
	Term_ (Record_, Enumerator_, Builtin_,
		Plan_, Exit_, Do_, Be_, Bind_, Coabstract_, Variable_, Let_, Error_),
	Statement_ (Cut_),
	Coterm_ (Case_, Selector_, Use_, Expose_,
		Covariable_, Abstract_, Function_, Colet_),
	term_names, get_term_name, coterm_names, get_coterm_name, stack_show, qshow, old_show,
	compress,
	BuiltinValue (BuiltinInteger, BuiltinString, BuiltinBoolean,
		BuiltinList, BuiltinColist),
	Expression_, map_, transformTerm_, transformCoterm_,
	update_origin) where
import List
import Maybe

import Basic

data Term_ =
	      Record_ [Term_]
	    | Enumerator_ Int Int Term_  -- Enumerator_ n_enums enum value
	    | Builtin_ BuiltinValue
	    | Plan_ Coterm_
	    | Exit_ String
	    | Do_ String String (IO Term_)
	    | Be_ String Coterm_
	    | Bind_ String String String Term_ Coterm_
	    | Error_ String
            | Coabstract_ VarName Statement_ (Maybe Int)  -- The VarName is entirely for display.
	    | Variable_ Int
	    | Let_ [(VarName, Either Term_ Coterm_, Maybe Int)] Term_  -- The VarNames are entirely for display.

data Statement_ =
	      Cut_ String Term_ Coterm_

data Coterm_ =
	      Case_ [Coterm_]
	    | Selector_ Int Int Coterm_  -- Selector_ n_options selection value
	    | Use_ Term_
	    | Expose_ Coterm_ (Integer, String) (Maybe Integer)
			-- (id, shown expression) (maybe next-id)
	    | Covariable_ Int
	    | Abstract_ VarName Statement_ (Maybe Int) -- The VarName is entirely for display.
	    | Function_ String String (Term_ -> Coterm_)
	    | Colet_ [(VarName, Either Term_ Coterm_, Maybe Int)] Coterm_  -- The VarNames are entirely for display.

-- Let_ and Colet_ provide a "stack" which should satisfy all variable
-- references in the term/coterm that they accompany.

instance Show Term_ where
    show x = case x of
	Coabstract_ str stmt stack_size -> "(" ++ show stmt ++ "(" ++ show stack_size ++ "))." ++ str
	Builtin_ v -> show v
	Record_ [] -> "<>rec"
	Record_ (t:ts) -> "<" ++ show t ++ concatMap (\s -> "," ++ show s) ts ++ ">rec"
	Enumerator_ m n t -> "<" ++ show t ++ ">in" ++ show n
	Plan_ c -> "[" ++ show c ++ "]pln"
	Exit_ _ -> "xit"
	Do_ _ name c -> "<" ++ name ++ ">do"
	Be_ _ c -> "[" ++ show c ++ "]be"
	Bind_ _ _ _ deliverer receiver -> "<" ++ show deliverer ++ ";" ++ show receiver ++ ">"
	Error_ _ -> "err"
	Variable_ num -> "Variable_:" ++ show num
	Let_ e t -> "(let " ++ show e ++ " in " ++ show t ++ ")"

instance Show Statement_ where
    show x = case x of
	Cut_ origin v c -> show v ++ "$" ++ show c

instance Show Coterm_ where
    show x = case x of
	Case_ [] -> "cas[]"
	Case_ (c:cs) -> "cas[" ++ show c ++ concatMap (\c-> "," ++ show c) cs ++ "]"
	Selector_ m n c -> show n ++ suffix ++ "[" ++ show c ++ "]"
	    where
		suffix = case n `mod` 10 of
		    1 -> "st"
		    2 -> "nd"
		    3 -> "rd"
		    otherwise -> "th"
	Use_ t -> "use<" ++ show t ++ ">"
	Expose_ c _ _ -> "exp[" ++ show c ++ "]"
	Covariable_ num -> "Covariable_:" ++ show num
	Abstract_ var stmt stack_size -> var ++ ".[" ++ show stmt ++ "[" ++ show stack_size ++ "]]"
	Function_ _ name f -> "bif:" ++ name
	Colet_ e c -> "[colet " ++ show e ++ " in " ++ show c ++ "]"

stack_show :: Bool -> String -> [(VarName, Either Term_ Coterm_, Maybe Int)] -> String
stack_show quick prefix es = if null es then prefix ++ "<<empty>>\n"
			else unlines $ zipWith liner [0..] limited_es
    where
	numw = length (show (length es)) + 1
	liner n e = prefix ++ reverse (take numw (reverse (show n) ++ repeat ' ')) ++ " "
			   ++ (if quick then qeshow else eshow) (prefix ++ replicate (numw+1) ' ') e
	eshow prefix (nm,e) = nm ++ ":" ++ case e of
		Left (Let_ e t) -> if null e then "let [] in " ++ show t
					else "let\n" ++ stack_show quick prefix e ++ prefix ++ "in " ++ show t
		Right (Colet_ e c) -> if null e then "colet [] in " ++ show c
					else "colet\n" ++ stack_show quick prefix e ++ prefix ++ "in " ++ show c
		Left t -> show t
		Right c -> show c
	qeshow prefix (nm,e) = nm ++ ": " ++ case e of
		Left t -> qshow t
		Right c -> qshow c
	limited_es = limit es Nothing
	limit ((a,b,local_lim):more) lim = case new_lim of
		Just 0 -> []
		otherwise -> (a,b):(limit more (fmap (+ (-1)) new_lim))
	    where
		new_lim = case lim of
		    Just n -> case local_lim of
			    Just m -> Just (min m n)
			    Nothing -> Just n
		    Nothing -> local_lim
	limit [] lim = []
		
class QShow t where
    qshow :: t -> String
instance QShow Term_ where
    qshow x = case x of
	Coabstract_ str stmt stack_size -> "(" ++ qshow stmt ++ ")." ++ str
	Builtin_ v -> show v
	Record_ [] -> "<>rec"
	Record_ (t:ts) -> "<" ++ qshow t ++ concatMap (\s -> "," ++ qshow s) ts ++ ">rec"
	Enumerator_ m n t -> "<" ++ qshow t ++ ">in" ++ show n
	Plan_ c -> "[" ++ qshow c ++ "]pln"
	Exit_ _ -> "xit"
	Do_ _ name c -> "<" ++ name ++ ">do"
	Be_ _ c -> "[" ++ qshow c ++ "]be"
	Bind_ _  _ _ deliverer receiver -> "<" ++ qshow deliverer ++ ";" ++ qshow receiver ++ ">"
	Error_ _ -> "err"
	Variable_ num -> "Variable_:" ++ show num
	Let_ e t -> qshow t
instance QShow Statement_ where
    qshow x = case x of
	Cut_ origin v c -> qshow v ++ "$" ++ qshow c
instance QShow Coterm_ where
    qshow x = case x of
	Case_ [] -> "cas[]"
	Case_ (c:cs) -> "cas[" ++ qshow c ++ concatMap (\c-> "," ++ qshow c) cs ++ "]"
	Selector_ m n c -> show n ++ suffix ++ "[" ++ qshow c ++ "]"
	    where
		suffix = case n `mod` 10 of
		    1 -> "st"
		    2 -> "nd"
		    3 -> "rd"
		    otherwise -> "th"
	Use_ c -> "use<" ++ qshow c ++ ">"
	Expose_ c _ _ -> "exp[" ++ qshow c ++ "]"
	Covariable_ num -> "Covariable_:" ++ show num
	Abstract_ var stmt stack_size -> var ++ ".[" ++ qshow stmt ++ "]"
	Function_ _ name f -> "bif:" ++ name
	Colet_ e c -> qshow c

class Old_Show t where
    old_show :: [(VarName, Maybe (Either Term_ Coterm_))] -> t -> String
instance Old_Show Term_ where
    old_show stk x = case x of
	Coabstract_ str stmt stack_size -> "(" ++ old_show ((str,Nothing):stk) stmt ++ ")." ++ str
	Builtin_ v -> show v
	Record_ [] -> "<>rec"
	Record_ (t:ts) -> "<" ++ old_show stk t ++ concatMap (\s -> "," ++ old_show stk s) ts ++ ">rec"
	Enumerator_ m n t -> "<" ++ old_show stk t ++ ">in" ++ show n
	Plan_ c -> "[" ++ old_show stk c ++ "]pln"
	Exit_ _ -> "xit"
	Do_ _ name c -> "<" ++ name ++ ">do"
	Be_ _ c -> "[" ++ old_show stk c ++ "]be"
	Bind_ _ _ _ deliverer receiver -> "<" ++ old_show stk deliverer ++ ";" ++ old_show stk receiver ++ ">"
	Error_ _ -> "err"
	Variable_ num -> maybe
			("Variable_:" ++ show num)
			(\(name,e) -> case e of
				    Just ee -> case ee of
					    Left term -> old_show stk term
					    Right _ -> "Error: Variable_:" ++ show num ++ " refers to a covalue."
				    Nothing -> "Variable_:" ++ name)
			(lookup num (zip [0..] stk))
	Let_ e t -> old_show (map (\(n,tm,lim) -> (n, Just tm)) e) t
instance Old_Show Statement_ where
    old_show stk x = case x of
	Cut_ origin w c -> old_show stk w ++ "$" ++ old_show stk c
instance Old_Show Coterm_ where
    old_show stk x = case x of
	Case_ [] -> "cas[]"
	Case_ (c:cs) -> "cas[" ++ old_show stk c ++ concatMap (\c-> "," ++ old_show stk c) cs ++ "]"
	Selector_ m n c -> show n ++ suffix ++ "[" ++ old_show stk c ++ "]"
	    where
		suffix = case n `mod` 10 of
		    1 -> "st"
		    2 -> "nd"
		    3 -> "rd"
		    otherwise -> "th"
	Use_ c -> "use<" ++ old_show stk c ++ ">"
	Expose_ c _ _ -> "exp[" ++ old_show stk c ++ "]"
	Covariable_ num -> maybe
			("Covariable_:" ++ show num)
			(\(name,e) -> case e of
				    Just ee -> case ee of
					    Left term -> "Error: Covariable_:" ++ show num ++ " refers to a value."
					    Right coterm -> old_show stk coterm
				    Nothing -> "Covariable_:" ++ name)
			(lookup num (zip [0..] stk))
	Abstract_ var stmt stack_size -> var ++ ".(" ++ old_show ((var,Nothing):stk) stmt ++ ")"
	Function_ _ name f -> "bif:" ++ name
	Colet_ e c -> old_show (map (\(n,tm,lim) -> (n, Just tm)) e) c

class Compress t where
    compress :: Bool -> [(VarName, Maybe (Either Term_ Coterm_))] -> t -> (Integer, t)
instance Compress Term_ where
    compress real stk x = case x of
	Coabstract_ str stmt stack_size ->
			let (sz, stmt') = compress real ((str,Nothing):stk) stmt
			in (sz + 1, Coabstract_ str stmt' (Just 0))
	Builtin_ v -> (1, Builtin_ v)
	Record_ ts ->
			let (szs, ts') = unzip (map (compress real stk) ts)
			in (sum szs + 1, Record_ ts')
	Enumerator_ m n t ->
			let (sz, t') = compress real stk t
			in (sz + 1, Enumerator_ m n t')
	Plan_ c ->
			let (sz, c') = compress real stk c
			in (sz + 1, Plan_ c')
	Exit_ orig -> (1, Exit_ orig)
	Do_ orig name t -> (1, Do_ orig name t)
	Be_ orig c ->
			let (sz, c') = compress real stk c
			in (sz + 1, Be_ orig c')
	Bind_ orig opd1str opd2str deliverer receiver ->
			let
				(sz1, d') = compress real stk deliverer
				(sz2, r') = compress real stk receiver
			in (sz1 + sz2 + 1, Bind_ orig opd1str opd2str d' r')
	Error_ msg -> (1, Error_ msg)
	Variable_ num -> maybe
			(1, Variable_ num)
			(\(name,e) -> case e of
				    Just ee | real -> case ee of
					    Left term -> compress real stk term
					    Right _ -> error ("Variable_:" ++ show num ++ " refers to a covalue.")
				    otherwise -> (1, Variable_ num))
			(lookup num (zip [0..] stk))
	Let_ e t -> if real then compress real (map (\(n,tm,lim) -> (n, Just tm)) e) t
			else let
				(szs1, e') = unzip (map (compress real []) (map (\(n,tm,lim) -> (n, Just tm)) e))
				(sz2, t') = compress real (map (\(n,tm,lim) -> (n, Just tm)) e) t
				in (sum szs1 + sz2 + 1, Let_ e t')
instance Compress Statement_ where
    compress real stk x = case x of
	Cut_ origin w c -> let
			(sz1, w') = compress real stk w
			(sz2, c') = compress real stk c
			in (sz1 + sz2 + 1, Cut_ origin w' c')
instance Compress ([(VarName, Either Term_ Coterm_, Maybe Int)], Statement_) where
    compress real ignore_stk (stk, stmt@(Cut_ origin w c)) = case real of
	    True -> let
			(sz, stmt') = compress real stk' stmt
			in (sz, ([], stmt'))
	    False -> let
			(sz1, w') = compress real stk' (Let_ stk w)
			(sz2, c') = compress real stk' (Colet_ stk c)
			in (sz1 + sz2, (stk, stmt))			
	where
	    stk' = map (\(n, tm, _) -> (n, Just tm)) stk
instance Compress Coterm_ where
    compress real stk x = case x of
	Case_ cs -> let
			(szs, cs') = unzip (map (compress real stk) cs)
			in (sum szs + 1, Case_ cs')
	Selector_ m n c -> let
			(sz, c') = compress real stk c
			in (sz + 1, Selector_ m n c')
	Use_ t -> let
			(sz, t') = compress real stk t
			in (sz + 1, Use_ t')
	Expose_ c id next -> let
			(sz, c') = compress real stk c
			in (sz + 1, Expose_ c' id next)
	Covariable_ num -> maybe
			(1, Covariable_ num)
			(\(name,e) -> case e of
				    Just ee | real -> case ee of
					    Left term -> error ("Covariable_:" ++ show num ++ " refers to a value.")
					    Right coterm -> compress real stk coterm
				    otherwise -> (1, Covariable_ num))
			(lookup num (zip [0..] stk))
	Abstract_ var stmt stack_size -> let
			(sz, stmt') = compress real ((var,Nothing):stk) stmt
			in (sz + 1, Abstract_ var stmt' (Just 0))
	Function_ orig name f -> (1, Function_ orig name f)
	Colet_ e c -> if real then compress real (map (\(n,tm,lim) -> (n, Just tm)) e) c
			else let
				(szs1, e') = unzip (map (compress real []) (map (\(n,tm,lim) -> (n, Just tm)) e))
				(sz2, c') = compress real (map (\(n,tm,lim) -> (n, Just tm)) e) c
				in (sum szs1 + sz2 + 1, Colet_ e c')
instance Compress (VarName, Maybe (Either Term_ Coterm_)) where
    compress real stk (nm, mtc) = case mtc of
		Just (Left t) -> let
			(sz, t') = compress real stk t
			in (sz + 1, (nm, Just (Left t')))
		Just (Right c) -> let
			(sz, c') = compress real stk c
			in (sz + 1, (nm, Just (Right c')))
		Nothing -> (1, (nm, Nothing))

term_names = tail $ listsOf $ reverse ['a'..'z']
get_term_name names = head (term_names \\ names)

coterm_names = tail $ listsOf ['a'..'z']
get_coterm_name names = head (coterm_names \\ names)

class Expression_ e where
	transformSubexpressions_ :: (forall t . (Expression_ t) => (t -> t)) -> e -> e
	foldExpression_ :: (forall t . (Expression_ t) => (t -> [x] -> x)) -> e -> x
	transformTerm_ :: (Term_ -> Term_) -> e -> e
	transformCoterm_ :: (Coterm_ -> Coterm_) -> e -> e

data AnExpression = forall e. (Expression_ e) =>  AnExpression e

map_ :: Expression_ e => (forall t . (Expression_ t) => (t -> t)) -> e -> e
map_ fn = fn . transformSubexpressions_ (map_ fn)

instance Expression_ Term_
    where
	transformSubexpressions_ fn t = case t of
		Record_ ts -> Record_ $ map fn ts
		Enumerator_ m n t -> Enumerator_ m n $ fn t
		Plan_ l -> Plan_ $ fn l
		Coabstract_ name stmt stack_size -> Coabstract_ name (fn stmt) stack_size
		Builtin_ (BuiltinList l) -> Builtin_ (BuiltinList (map fn l))
		Builtin_ val -> t
		Exit_ orig -> t
		Do_ _ name act -> t
		Be_ orig c -> Be_ orig $ fn c
		Bind_ orig opd1str opd2str deliverer receiver -> Bind_ orig opd1str opd2str (fn deliverer) (fn receiver)
		Error_ msg -> t
		Variable_ str -> t
		Let_ es t -> Let_ (map efn es) (fn t)
		    where
			efn (nm, e, lim) = (nm, case e of
				Left t -> Left (fn t)
				Right c -> Right (fn c),
			    lim)
	foldExpression_ f t = f t foldeds
	    where
		foldeds = case t of
		    Record_ ts -> map (foldExpression_ f) ts
		    Enumerator_ m n t -> [foldExpression_ f t]
		    Plan_ l -> [foldExpression_ f l]
		    Coabstract_ name stmt stack_size -> [foldExpression_ f stmt]
		    Builtin_ (BuiltinList l) -> map (foldExpression_ f) l
		    Builtin_ val -> []
		    Exit_ _ -> []
		    Do_ _ name act -> []
		    Be_ _ c -> [foldExpression_ f c]
		    Bind_ _ _ _ deliverer receiver -> [foldExpression_ f deliverer, foldExpression_ f receiver]
		    Error_ _ -> []
		    Variable_ str -> []
		    Let_ es t -> foldExpression_ f t : map eff es
			where
			    eff (nm, e, lim) = case e of
					Left t -> foldExpression_ f t
					Right c -> foldExpression_ f c
	transformTerm_ fn t = fn t
	transformCoterm_ fn c = c

instance Expression_ Coterm_
    where
	transformSubexpressions_ fn c = case c of
		Case_ cs -> Case_ $ map fn cs
		Selector_ m n c -> Selector_ m n (fn c)
		Use_ t -> Use_ (fn t)
		Expose_ c id next -> Expose_ (fn c) id next
		Covariable_ name -> c
		Abstract_ str stmt stack_size -> Abstract_ str (fn stmt) stack_size
		Function_ orig name f -> c
		Colet_ es c -> Colet_ (map efn es) (fn c)
		    where
			efn (nm, e, lim) = (nm, case e of
				Left t -> Left (fn t)
				Right c -> Right (fn c),
			    lim)
	foldExpression_ f c = f c foldeds
	    where
		foldeds = case c of
		    Case_ cs -> map (foldExpression_ f) cs
		    Selector_ m n c -> [foldExpression_ f c]
		    Use_ t -> [foldExpression_ f t]
		    Expose_ c id next -> [foldExpression_ f c]
		    Covariable_ name -> []
		    Abstract_ str stmt stack_size -> [foldExpression_ f stmt]
		    Function_ orig name f -> []
		    Colet_ es c -> foldExpression_ f c : map eff es
			where
			    eff (nm, e, lim) = case e of
					Left t -> foldExpression_ f t
					Right c -> foldExpression_ f c
	transformTerm_ fn c = c
	transformCoterm_ fn c = fn c

instance Expression_ Statement_ where
	transformSubexpressions_ fn (Cut_ origin term coterm) = Cut_ origin (fn term) (fn coterm)
	foldExpression_ f stmt@(Cut_ origin tm cotm)
		= f stmt [foldExpression_ f tm, foldExpression_ f cotm]
	transformTerm_ fn stmt = stmt
	transformCoterm_ fn stmt = stmt

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --

-- Each variant of BuiltinValue represents a distinct type of the
-- target language.
data BuiltinValue =
	  BuiltinInteger Integer
	| BuiltinString String
	| BuiltinBoolean Bool
	| BuiltinList [Term_]      -- only used in Builtin_
	| BuiltinColist [Coterm_]  -- only used in Builtin_

instance Show (BuiltinValue) where
    show x = case x of
		BuiltinInteger i -> show i
		BuiltinList l -> case l of
			[] -> "[]"
			(m:ms) -> "[" ++ show m ++ concat (map (\m -> " " ++ show m) ms) ++ "]"
		BuiltinBoolean tf -> if tf then "true" else "false"
		BuiltinString str -> "\"" ++ str ++ "\""
		BuiltinColist l -> case l of
			[] -> "[]"
			(m:ms) -> "[" ++ show m ++ concat (map (\m -> " " ++ show m) ms) ++ "]"

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --

update_term_origin :: String -> Term_ -> Term_
update_term_origin orig term = case term of
    Do_ _ name act -> Do_ orig name act
    Be_ _ cotrm -> Be_ orig cotrm
    Bind_ _ opd1str opd2str trm cotrm -> Bind_ orig opd1str opd2str trm cotrm
    Exit_ _ -> Exit_ orig
    otherwise -> term
update_coterm_origin :: String -> Coterm_ -> Coterm_
update_coterm_origin orig coterm = case coterm of
    Function_ _ name f -> Function_ orig name f
    otherwise -> coterm
update_origin :: Expression_ e => String -> e -> e
update_origin orig = transformTerm_ (update_term_origin orig) . transformCoterm_ (update_coterm_origin orig)