{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Lower (lowers, create_stack,
	byValueFunction, byValueApplication,
	byValueCoapplication, byValueConstantFunction,
	byNameFunction, byNameApplication, byNameConstantFunction,
	Stack(),
	push_gapvariable, stack_size, var_at, covar_at, push_variable, push_covariable
    ) where

import List hiding (insert)
import qualified Data.List
import Maybe
import Data.Map hiding ((\\), map)
import Control.Monad

import Basic
import Position
import Kind
import Type
import AST
import Parameters
import Intermediate

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- Semantic Analysis: Lowering
-- Semantic analysis consists of binding followed by lowering.
-- Lowering is necessary because the AST will include lambda abstractions
-- and function applications, whereas we want evaluation to operate
-- at a more basic level, using the only the BoundAbstractTerm and
-- BoundAbstractCoterm operators.

{-
The lowering functions generally accept a Term/Coterm and yield a Term_/Coterm_.
They operate recursively, which would be straightforward in itself.  One
twist is that lowering binds the I/O functions to files and therefore it
is part of the interpreter's IO, and hence is in a monad.

A more subtle twist is that the several functions which operate on Term_s
and Coterm_s depend on the simulated stack when assigning offsets to
variables.  Therefore, none of these functions accepts a straight Term_
but rather a (Stack -> m Term_) which gets fixed up in the context which
has the complete stack.
-}

{- 
Lower_bind yields an action term which, when evaluated in juxtaposition with
the appropriate coterm, (1) executes 'act1' to a term, (2) combines the term
with the function 'bind2' to find another action, and (3) executes the second action
and passes the resulting value to the coterm.
Because step (2) involves 'evaluate', it requires context.  Not only that, but it needs
to yield changed context.

However, (A) 'evaluate' always goes to the bitter end and does not return stuff,
and (B) I stumbled on such attempts earlier.  So the only possible approach would be
for 'bind' to return an expression which, during evaluation, has side-effects as desired.
-}

{- At such time as this expression is evaluated, it applies the
   function bind2 to the argument ([iocot]act * use<[cont]pln>).cont.
   It will substitute the argument into the function, then
   evaluate the resulting action term, which is rigged to evaluate
      [iocot]act * use<[cont]pln>
   Where 'cont' is a continuation that will pass along the evaluated result of the action.
   The action is effected right away!
      [cont]pln * use<retterm>    (Iocot is expected to return a coterm which immediately encapsulates a term.)
      retterm * cont              Retterm is evaluated and passed along to cont.
-}

lower_bind :: (Monad m) => Span -> Term -> Term -> (Stack -> m Term_) -> (Stack -> m Term_) -> Stack -> m Term_
lower_bind spn (TM (_,spn0)_) (TM (_,spn1)_) fact1 fbind2 stk = do
	-- Bind arg [z->func$[(z)]]   z:s&~!t  func:~(s&~!t)  arg:!s
	act1 <- fact1 stk
	bind2 <- fbind2 stk' -- ~(s & ~m t)  [z->bind2$[(z)]]
	return $ bindterm act1 bind2
    where
		stk' = push_variable stk "@z"
		bindterm act1 bind2 = Bind_ (show spn) (show spn0) (show spn1) act1 (Abstract_ "@z"
							    (Cut_ explanation bind2 (Use_ bterm))
							    (Just (stack_size stk'))
						 )
		bterm = get_term stk' "@z"
		explanation = "passing the result of " ++ show spn0 ++ " to " ++ show spn1
{-

	        byValueApplication spn "bind" act1' bind2
    where
	-- Strip layers from act1.
	act1' stk = do
		act1 <- fact1 stk'
		return $ Coabstract_ "@a" (Cut_ act1 (Use_ (Plan_ (get_coterm stk' "@a")))) (Just (stack_size stk'))
	    where
		stk' = push_covariable stk "@a"
-}

lower_cobind :: (Monad m) => Span -> Coterm -> Coterm -> MonadModel -> (Stack -> m Coterm_) -> (Stack -> m Coterm_) -> Stack -> m Coterm_
lower_cobind spn (TO (_,spn0)_) (TO (_,spn1)_) (UseStrictMonad strict_monad) farg ffunc stk = do
	-- x.(<([a]pln * arg).a,x>par * func):!~t  x:!~t  func:!~s&~t  a:~s  arg:!~s
	-- [x->([z->(&z&x&)$func ])$ arg]  x:t  z:s  func:s&t  arg:~s
	-- [x->([z->x$[(Bind z func)]])$arg]   x:~!t  z:!s  func:s&~!t  arg:~!s
	arg <- farg stk'
	func <- (if strict_monad then adjust_function else id) ffunc stk''
	return $ first_go func arg
    where
	stk' = push_variable stk "@x"
	stk'' = push_variable stk' "@z"
	first_go func arg = Abstract_ "@x"
		(Cut_ explanation (functerm func) arg)
		(Just (stack_size stk'))
	functerm func = Plan_ $ Abstract_ "@z"
					(Cut_ "lower_cobind"
					    contterm
					    (Use_ $ Bind_ (show spn) (show spn0) (show spn1) argterm func)
					)
					(Just (stack_size stk''))
	contterm = get_term stk'' "@x"
	argterm = get_term stk'' "@z"
	explanation = "passing the result of " ++ show spn0 ++ " to " ++ show spn1

	-- The function passed in is a usual by-name thing func:~~s&~!t.
	-- Adjust it.  [&x&c&->(&([(x)])&c&)$func]
	--             [y->y$0th[x->y$1st[c-><[use<x>]pln,c>rec$func]]]
	adjust_function f stk =  do
		f''' <- f stk'''
		return $ adjusted f'''
	    where
		stk' = push_variable stk "@y"
		stk'' = push_variable stk' "@x"
		stk''' = push_variable stk'' "@c"
		adjusted func = Abstract_ "@y"
				(Cut_ "lower_cobind adjustment" y' xselector)
				(Just (stack_size stk'))
		    where
			y' = get_term stk' "@y"
			xselector = Selector_ 2 0 xabstraction
			xabstraction = Abstract_ "@x"
					(Cut_ "lower_cobind adjustment" y'' cselector)
					(Just (stack_size stk''))
			y'' = get_term stk'' "@y"
			cselector = Selector_ 2 1 cabstraction
			cabstraction = Abstract_ "@c"
					(Cut_ "lower_cobind adjustment" funccontext func)
					(Just (stack_size stk'''))
			funccontext = Record_ [Plan_ (Use_ x'''), c''']
			x''' = get_term stk''' "@x"
			c''' = get_term stk''' "@c"

-- == -- ==-- == -- ==-- == -- ==-- == -- ==-- == -- ==-- == -- ==-- == -- ==
-- In the following functions, the Int and Map represent a stack that
-- holds values/covalues of names defined in an outer scope.  The Int is the
-- number of items on the stack, while the Map associates names with
-- either values or locations.  If the associated term is a Variable_ with
-- a negative offset, then the value will be found on the runtime stack
-- at that offset from the root of the stack. Otherwise the value is a builtin.
-- The stack also accomodates the representation of covariables as forcing
-- expressions, e.g. Abstract_ "y" (Cut_ (Variable_ -1) (Use_ (Variable 0)))
--                   [y->x$[(y)]]
-- where 'x' holds the thunk.
data Stack = Stack Int (Map VarName (Either Term_ Coterm_))
create_stack :: (Map VarName (Either Term_ Coterm_)) -> Stack
create_stack map = Stack 0 map
get_term :: Stack -> VarName -> Term_
get_term stk vname = maybe (error $ "Variable '" ++ vname ++ "' is not in scope.")
					id
					(lookup_term stk vname)
get_coterm :: Stack -> VarName -> Coterm_
get_coterm stk vname = maybe (error $ "Covariable '" ++ vname ++ "' is not in scope.")
					id
					(lookup_coterm stk vname)
lookup_term :: Stack -> VarName -> Maybe Term_
lookup_term stk@(Stack n tmap) vname = maybe Nothing
					(\e -> case e of
					    Left t -> Just (map_ (transformTerm_ (stackToDeBruijn stk)) t)
					    Right x -> error $ "Covariable '" ++ vname ++ "' used as a variable.")
					(Data.Map.lookup vname tmap)
lookup_coterm :: Stack -> VarName -> Maybe Coterm_
lookup_coterm stk@(Stack n tmap) vname = maybe Nothing
					(\e -> case e of
					    Right (Covariable_ n) | n < 0 -> Just $ covar_at n stk
					    Right c -> Just (map_ (transformTerm_ (stackToDeBruijn stk)) c)
					    Left x -> error $ "Variable '" ++ vname ++ "' used as a covariable.")
					(Data.Map.lookup vname tmap)
stack_size :: Stack -> Int
stack_size (Stack n _) = n
push_variable :: Stack -> VarName -> Stack
push_variable (Stack nvar tmap) tname = Stack (nvar+1) (insert tname (Left $ Variable_ (-(nvar+1))) tmap)
push_covariable :: Stack -> VarName -> Stack
push_covariable (Stack nvar tmap) cname = Stack (nvar+1) (insert cname (Right $ Covariable_ (-(nvar+1))) tmap)
push_gapvariable :: Stack -> Stack
push_gapvariable (Stack nvar tmap) = Stack (nvar+1) tmap

{-
 On the simulated stack, Variable_ and Covariable_ have an integer offset indicating the
 "absolute address" where the value will be found, a negative number no larger than the
 stack size.  -1 indicates the stack root, not the top were the stack grows.
 Lowering looks up a name and creates a Variable_ or Covariable_ with the corresponding
 nonnegative offset, calculated based on the stack size. The top of the stack will be at
 an offset of 0.
-}
-- Nvar is an "absolute address".
var_at :: Int -> Stack -> Term_
var_at nvar stk = Variable_ (nvar + stack_size stk)
covar_at :: Int -> Stack -> Coterm_
covar_at nvar stk = Covariable_ (nvar + stack_size stk)

dump_stack :: Stack -> String
dump_stack = show . sort_stack
sort_stack :: Stack -> [String]
sort_stack stk@(Stack nvar tmap) = stack_es
    where
	es = toList tmap
	mstack_es = map (\(nm, elem) -> case elem of
						Left (Variable_ n) | n < 0 -> Just (n + nvar, nm)
						Right (Covariable_ n) | n < 0 -> Just (n + nvar, nm)
						_ -> Nothing)
		        es
	stack_es = map (\n -> maybe "" id (Data.List.lookup n (concatMap maybeToList mstack_es))) [0 .. nvar-1]

stackToDeBruijn :: Stack -> Term_ -> Term_
stackToDeBruijn stk t = case t of
	(Variable_ n) | n < 0 -> var_at n stk
	otherwise -> t

-- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- ==

lowers params m_handle = (lower_term, lower_coterm)
  where
    tl = trace_lowering params
    mm = monad_model params

    verbose = if tl then m_handle else Nothing

    show' :: (TShow t) => t -> String
    show' = tshow ShowSpan True True
    lower_term :: Term -> Stack -> IO Term_
    lower_term ttrm@(TM (typ@(Type k _ tp),spn) trm) stk = do
	aside verbose (strictList_ $ "lowering term " ++ tshow TopShowSpan False False ttrm)
	aside verbose (strictList_ $ "  stack: " ++ dump_stack stk)
		
	ltrm <- case trm of
	    ListTerm ts -> do
			lts <- mapM (flip lower_term stk) ts
			return $ Builtin_ $ BuiltinList lts
	    VariableTerm vname -> return $ map_ (update_origin (show spn)) (get_term stk vname)
	    IntegerTerm n -> return $ Builtin_ $ BuiltinInteger n
	    StringTerm str -> return $ Builtin_ $ BuiltinString str
	    SeqTerm prelim result ->
			lower_bind spn prelim result (lower_term prelim) (fval $ lower_term result) stk
		where
		    fval r = byValueConstantFunction (show spn ++ " fval") "const func" r
	    BindTerm act binder ->
			lower_bind spn act opd2 (lower_term act) (lower_term binder) stk
		where
		    opd2 = case binder of
				TM ltp (LambdaTerm pat m) -> m
				otherwise -> binder
	    ApplicationTerm arg func ->
			byValueApplication spn (show' arg ++ "#" ++ show' func)
							 (lower_term arg) (lower_term func) stk
	    VoidTerm -> return (Record_ [])
	    ProductTerm ts -> do
			l_ts <- mapM (flip lower_term stk) ts
			return $ Record_ l_ts
	    PlanTerm c -> do
			l_c <- lower_coterm c stk
			return $ Plan_ l_c
	    LambdaTerm pattern@(Pattern ptp pat) t -> do
		    case pat of
			NamePat vname -> byValueFunction (show spn) "lambda" l_t stk
			    where
				l_t stk = lower_term t stk'
				    where
					stk' = push_variable stk vname
			otherwise ->
				byValueFunction (show spn) "lambda" castmt' stk
			    where
				castmt' stk = do
					let lstmt = lower_statement spn (t, TO (noT,no_span) $ CovariableTerm cname)
					stmt <- lower_pattern' spn pattern lstmt (pattern_handle, stk')
					return $ Coabstract_ cname stmt (Just (stack_size stk'))
				    where
					(cname:_) = unusedOf t \\ usedOf pat
					(pattern_handle, stk'') = push_pattern_handle pattern stk
					stk' = push_covariable stk'' cname
				
	    AbstractTerm copat t c -> lower_copattern spn copat (lower_statement spn (t, c)) stk
	    AnnotatedTerm _ t -> lower_term t stk
	    GenericTypeTerm k nm t -> lower_term t stk
	aside verbose $ "lowered term " ++ tshow TopShowSpan False False ttrm ++ "\nto " ++ show ltrm
	aside verbose $ "   a.k.a. " ++ old_show (map (\n -> (n,Nothing)) (sort_stack stk)) ltrm
	return ltrm

    lower_coterm :: Coterm -> Stack -> IO Coterm_
    lower_coterm ttrm@(TO (typ@(Type k _ tp),tsp) trm) stk = do
	aside verbose $ "lowering coterm " ++ tshow TopShowSpan False False ttrm
	aside verbose $ "  stack: " ++ dump_stack stk

	lcot <- case trm of
	    ColistTerm ts -> die "colists not supported"
	    CovariableTerm nm -> return $ map_ (update_origin (show tsp)) (get_coterm stk nm)
	    -- CobindTerm act1 (TO _ (ColambdaTerm (Copattern _ pat) act2)))
	    CobindTerm act subsequent ->
			lower_cobind tsp act subsequent mm (lower_coterm act) (lower_coterm subsequent) stk
	    CoseqTerm act1 act2 ->
			lower_cobind tsp act1 act2 mm (lower_coterm act1) (fval $ lower_coterm act2) stk
		where
		    fval l_act2 = byNameConstantFunction (show tsp) "const~func" l_act2
	    CoapplicationTerm arg func ->
			byNameApplication tsp (show' arg ++ "~#" ++ show' func)
							(lower_coterm arg) (lower_coterm func) stk
	    UseTerm tm -> do
			l_tm <- lower_term tm stk
			return $ Use_ l_tm
	    UnreachedTerm -> return $ Case_ []
	    SumTerm cs -> do
			l_cs <- mapM (flip lower_coterm stk) cs
			return $ Case_ l_cs
	    ColambdaTerm copattern@(Copattern ptp copat) c -> do
		    case copat of
			NameCopat vname -> byNameFunction (show tsp) "~lambda" l_c stk
			    where
				l_c stk = lower_coterm c stk'
				    where
					stk' = push_covariable stk vname
			otherwise -> byNameFunction (show tsp) "~lambda" astmt' stk
			    where
				astmt' stk = do
					let lstmt = lower_statement tsp (TM (noT,no_span) (VariableTerm tname),
								     c)
					stmt <- lower_copattern' tsp copattern lstmt
								 (copattern_handle, stk')
					return $ Abstract_ tname stmt (Just (stack_size stk'))
				    where
					(tname:_) = unusedOf c \\ usedOf copat
					(copattern_handle, stk'') = push_copattern_handle copattern stk
					stk' = push_variable stk'' tname

	    CoabstractTerm pat t c -> lower_pattern tsp pat (lower_statement tsp (t, c)) stk
	    CoannotatedTerm _ c -> lower_coterm c stk
	    AbstractTypeTerm k nm t -> lower_coterm t stk
	aside verbose $ "lowered " ++ tshow TopShowSpan False False ttrm ++ "\nto " ++ show lcot
	aside verbose $ "   a.k.a. " ++ old_show (map (\n -> (n,Nothing)) (sort_stack stk)) lcot
	return lcot

    lower_statement :: Span -> (Term, Coterm) -> Stack -> IO Statement_
    lower_statement spn (base_term, base_coterm) stk = do
	    base_term_ <- lower_term base_term stk
	    base_coterm_ <- lower_coterm base_coterm stk
	    return $ Cut_ (show spn) base_term_ base_coterm_

    lower_pattern :: Span -> Pattern -> (Stack -> IO Statement_) -> Stack -> IO Coterm_
    lower_pattern spn pattern lowerer stk = 
	let stk'@(vname,actual_stk') = push_pattern_handle pattern stk
	  in do
	    stmt <- lower_pattern' spn pattern lowerer stk'
	    return $ Abstract_ vname stmt (Just (stack_size actual_stk'))
    push_pattern_handle :: Pattern -> Stack -> (VarName, Stack)
    push_pattern_handle (Pattern ptp pat) stk = (pattern_handle, stk')
	where
	    stk' = push_variable stk pattern_handle
	    pattern_handle = case pat of
		    NamePat nm -> nm
		    otherwise -> "@" ++ show (stack_size stk)
    -- The pattern handle must be on the stack.
    lower_pattern' :: Span -> Pattern -> (Stack -> IO Statement_) -> (VarName, Stack) -> IO Statement_
    lower_pattern' spn (Pattern ptp pat) lowerer (vname, stk) = do
	    aside verbose (strictList_ $ "lowering pattern " ++ tshow TopShowSpan False False pat)
	    stmt <- case pat of
		NamePat nm -> lowerer stk
		VoidPat -> lowerer stk
		AndPat mpats -> do
			-- Adapt the term to accept vname of product type and
			-- pass along its components using the names in mpats.
			let mpats0 = zip [0..] mpats
			let m = length mpats
			let folded_lowerer = foldr (\(n,mpat)-> \lowerer ->
						maybe
						lowerer
						(\pat -> \stk -> do
							v <- return $! get_term stk vname
							cot0 <- lower_pattern spn pat lowerer stk
							return $ Cut_ (show spn) v (Selector_ m n cot0))
					        mpat)
					lowerer mpats0
			folded_lowerer stk
		NotPat copat -> do
		        ctrm0 <- do
				trm0 <- lower_copattern spn copat lowerer stk
				return (Use_ trm0)
			let stmt = Cut_ (show spn) (get_term stk vname) ctrm0
			return stmt
		SpecificPat stp pat -> do
			cot0 <- lower_pattern spn pat lowerer stk
			let stmt = Cut_ (show spn) (get_term stk vname) cot0
			return stmt
	    aside verbose $ "lowered pattern " ++ tshow TopShowSpan False False pat ++ " to " ++ show stmt
	    return stmt

    lower_copattern :: Span -> Copattern -> (Stack -> IO Statement_) -> Stack -> IO Term_
    lower_copattern spn copattern lowerer stk =
	let stk'@(vname,actual_stk') = push_copattern_handle copattern stk
	  in do
	    stmt <- lower_copattern' spn copattern lowerer stk'
	    return $ Coabstract_ vname stmt (Just (stack_size actual_stk'))
    push_copattern_handle :: Copattern -> Stack -> (VarName, Stack)
    push_copattern_handle (Copattern ptp pat) stk = (copattern_handle, stk')
	where
	    stk' = push_covariable stk copattern_handle
	    copattern_handle = case pat of
		    NameCopat nm -> nm
		    otherwise -> "@" ++ show (stack_size stk)
    lower_copattern' :: Span -> Copattern -> (Stack -> IO Statement_) -> (VarName, Stack) -> IO Statement_
    lower_copattern' spn (Copattern ptp pat) lowerer (vname, stk) = do
	    aside verbose (strictList_ $ "lowering copattern " ++ tshow TopShowSpan False False pat)
	    stmt <- case pat of
		NameCopat nm -> lowerer stk
		UnreachedCopat -> lowerer stk
		OrCopat mpats -> do
			-- Adapt the coterm to accept vname of product type and
			-- pass along its components using the names in mpats.
			let mpats0 = zip [0..] mpats
			let m = length mpats
			let folded_lowerer = foldr (\(n,mpat)-> \lowerer ->
						maybe
						lowerer
						(\pat -> \stk -> do
							v <- return $! get_coterm stk vname
							trm0 <- lower_copattern spn pat lowerer stk
							return $ Cut_ (show spn) (Enumerator_ m n trm0) v)
						mpat)
					lowerer mpats0
			folded_lowerer stk
		NotCopat pat -> do
			cot0 <- lower_pattern spn pat lowerer stk
			let stmt = Cut_ (show spn) (Plan_ cot0) (get_coterm stk vname)
			return stmt
		PackageCopat stp pat -> do
			trm0 <- lower_copattern spn pat lowerer stk
			let stmt = Cut_ (show spn) trm0 (get_coterm stk vname)
			return stmt
	    aside verbose $ "lowered copattern " ++ tshow TopShowSpan False False pat ++ " to " ++ show stmt
	    return stmt
	
-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --

-- The term argument ('body') refers to a Variable_ at location -1,
-- which becomes bound to create the abstraction.
-- The stack which is passed to fbody needs to be pushed with
-- a variable name in order to obtain the needed offsets.
byValueFunction :: (Monad m) => String -> String -> (Stack -> m Term_) -> Stack -> m Term_
byValueFunction orig name fbody stk = do
    	    -- [z.(z * fst[x.(z * snd[use<N>])])]pln
	    body <- fbody stk'
	    return $ Plan_ $ Abstract_ "@z" (stmt body) (Just (stack_size stk'))
	where
	    stk' = push_variable stk "@z"
	    stk'' = push_variable stk' "@x"
	    stmt body = Cut_ (orig ++ " selecting val") (get_term stk' "@z")
		    (Selector_ 2 0 $ Abstract_ "@x" (Cut_ (orig ++ " selecting context") (get_term stk'' "@z")
			(Selector_ 2 1 $ Use_ body)
		    ) (Just (stack_size stk'')))

byValueApplication :: (Monad m) => Span -> String -> (Stack -> m Term_) -> (Stack -> m Term_)
									-> Stack -> m Term_
-- (func * use<<arg,[a]pln>par>).a           :t    a:t   arg:s  func:~(s&~t)
byValueApplication name spn farg ffunc stk = do
	arg <- farg stk'
	func <- ffunc stk'
	let stmt = stmt_of arg func
	let stack_lim = Just (stack_size stk')
	return $ Coabstract_ "@a" stmt stack_lim
    where
	stk' = push_covariable stk "@a"
	stmt_of arg func = Cut_ (show spn) func app
	    where
		app =   -- use<<arg,[a]pln>par>
			Use_ (Record_ [arg, continuation_term])
		continuation_term = Plan_ $ get_coterm stk' "@a" -- was ??? Covariable_ 0

byValueCoapplication :: (Monad m) => Span -> (Stack -> m Coterm_) -> (Stack -> m Coterm_) -> Stack -> m Coterm_
byValueCoapplication spn farg ffunc stk = do
	arg <- farg stk'
	func <- ffunc stk'
	return $ Abstract_ "@z" (stmt arg func) (Just (stack_size stk'))
    where
	stk' = push_variable stk "@z"
	stmt arg func = Cut_ (show spn) app func
	    where
		app =           -- term by value, coterm by name<[func]not,value>
				Record_ [Plan_ arg, get_term stk' "@z"]

byValueConstantFunction :: (Monad m) => String -> String -> (Stack -> m Term_) -> Stack -> m Term_
byValueConstantFunction orig name body stk = byValueFunction orig name (\stk' -> body (push_gapvariable stk')) stk
		    
-- The coterm argument ('body') refers to a Covariable_ at location -1,
-- which becomes bound to create the abstraction.
-- The stack which is passed to fbody needs to be pushed with
-- a covariable name in order to obtain the needed offsets.
byNameFunction :: (Monad m) => String -> String -> (Stack -> m Coterm_) -> Stack -> m Coterm_
byNameFunction orig name fbody stk = do
	    -- x.(x * fst[use<(x * snd[N]).c>])   dual of (<[x.(<N>inr * c)]pln>inl * c).c
	    body <- fbody stk'
	    return $ Abstract_ "@x" (stmt body) (Just (stack_size stk'))
	where
	    stk' = push_variable stk "@x"
	    stk'' = push_covariable stk' "@c"
	    stmt body = (Cut_ orig
		    (get_term stk' "@x")
		    (Selector_ 2 argument $ 
			Use_ $
			    Coabstract_ "@c" (inner_stmt body)
					     (Just (stack_size stk''))
		    )
		)
	    inner_stmt body = (Cut_ orig
			    (get_term stk'' "@x")
			    (Selector_ 2 continuation body)
			)
	    (continuation, argument) = (1, 0)

byNameApplication :: (Monad m) => Span -> String -> (Stack -> m Coterm_)
						 -> (Stack -> m Coterm_) -> Stack -> m Coterm_
byNameApplication spn name farg ffunc stk = do
        -- (func * [use<arg>,a]).a    normal  [use<arg>,ctx]
        -- z.(<[arg]pln,z>par * func) complemented  
	arg <- farg stk'
	func <- ffunc stk'
	return $ Abstract_ "@z" (stmt arg func) (Just (stack_size stk'))
    where
	stk' = push_variable stk "@z"
	stmt arg func = Cut_ (show spn) app func
	    where
		app =     -- [a,use<arg>]
			  Record_ [Plan_ arg, continuation]
		continuation = get_term stk' "@z" -- was ??? Variable_ {-z-} 0

byNameConstantFunction :: (Monad m) => String -> String -> (Stack -> m Coterm_) -> Stack -> m Coterm_
-- If we went with
--   x.(x * fst[use<(x * snd[N]).unused>])
-- then it would never evaluate the argument, and evaluation of
-- the argument is just what we need to happen.
byNameConstantFunction orig name body stk = byNameFunction orig name (\stk' -> body (push_gapvariable stk')) stk
