{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
import System (getArgs)
import System.Directory (removeFile, doesFileExist)
import IO (Handle, IOMode(WriteMode), openFile, hClose, stdout, stderr, hPutStrLn)
import Control.Monad.State
import Data.IORef

import Basic
import Position
import Kind
import Type
import Unify
import AST
import TypedAST
import Parse
import Intermediate
import Builtin
import Lower
import Evaluate
import Parameters
import UTF8Input

-- == ## == -- == ## == -- == ## == -- == ## == -- == ## == -- == ## == --

run_string params = run_string'
  where
    vb = is_verbose params
    tt = trace_types params
    tu = trace_unification params
    uc = unify_continue params
    tc = check_types params
    mm = monad_model params
    uio = use_lower_IO_type params
    ufn = use_lower_function_types params
    ulimit = unify_limit params
    tk = trace_kind_unification params
    coerce = coerce_coterms params

    nonstandard_output = stderr
    (lower_term, lower_coterm) = lowers params (Just nonstandard_output)
    evaluate = evaluate' params (Just nonstandard_output)
    verbose = if vb then (Just nonstandard_output) else Nothing
    verbose_unify = if tu then (Just nonstandard_output) else Nothing

    show' :: (TShow t) => t -> String
    show' = tshow ShowSpan True True

    -- run, tester', tester'': the framework for the parse-execute machine
    run_string' :: FilePath -> OutFileProxy -> String -> IO ()
    run_string' fn outdest str = do
	let parses = parse (expression ufn coerce :: Parser (Either Term Coterm)) fn str
	let OutFileProxy open close reset message = outdest
	case parses of
	    Left err -> do
			    reset
			    print err
	    Right expr -> do
			    liftIO $ hPutStrLn (maybe nonstandard_output id verbose) $ "--- Syntax checked. ---"
			    case expr of
				Left term -> tester' outdest term
				Right coterm -> cotester' outdest coterm

    tester' :: OutFileProxy -> Term -> IO ()
    tester' outdest expr = run_with_typevars $ do
	case verbose of 
	    Just h -> do
		let shown_expr = tshow TopShowSpan False False expr
		liftIO $ hPutStrLn h $ "-- as parsed --\n" ++ shown_expr
	    Nothing -> return()
	let ev = substituteLibraryBuiltins uio ufn mm expr
	uv <- if tc then do
	    aside verbose "-- About to check types. --"
	    ev' <- case ev of
		TM (Type k sp NoT, _) op -> return $ TM (iOVT uio sp noT, no_span) op
		TM (Type k _ (ApplicationT (Type _ _ (NameT "IOV" _)) _), _) op -> return ev
		TM (Type k _ (ApplicationT (Type _ _ (NameT "IO" _)) _), _) op -> return ev
		otherwise -> die ("Expression is not an action:\n" ++ tshow TopShowSpan tt False ev)
	    XExpr uv <- unify verbose_unify tk uc ulimit mm uio ufn
				(types_of_native_builtins uio ufn mm) (XExpr ev')
	    liftIO $ hPutStrLn (maybe nonstandard_output id verbose) $ "--- Types checked. ---"
	    aside verbose $ strictList_ $ "to " ++ tshow NoShowSpan True tk uv
	    return uv
	  else do
	    aside verbose "-- Skipping, not checking types. --"
	    return ev
	liftIO $ tester'' outdest uv

    tester'' :: OutFileProxy -> {- Semval -} Term -> IO ()
    tester'' outdest expr = do
	    let OutFileProxy open close reset message = outdest
	    case verbose of
		Just h -> do
		    hPutStrLn h "The unsubstituted expression:"
		    hPutStrLn h $ tshow NoShowSpan False False expr
		    hPutStrLn h $ show' expr
		Nothing -> return ()
	    runtime_output_ref <- newIORef Nothing
	    pv <- lower_term expr (create_stack (nativeBuiltins uio ufn mm runtime_output_ref))
	    let nv = (\e -> Cut_ "run2ning the program" e run_coterm) pv
	    case verbose of
		Just h -> do
		    hPutStrLn h "The original expression:"
		    hPutStrLn h $ show nv
		    hPutStrLn h "--- Starting evaluation. ---"
		Nothing -> return ()
	    runtime_output <- open
	    writeIORef runtime_output_ref (Just runtime_output)
	    evaluate nv
	    writeIORef runtime_output_ref Nothing
	    close runtime_output
	    message
	    return ()

    cotester' :: OutFileProxy -> Coterm -> IO ()
    cotester' outdest expr = run_with_typevars $ do
	case verbose of
	    Just h-> do
		    let shown_expr = tshow TopShowSpan False False expr
		    liftIO $ hPutStrLn h $ "-- as parsed --\n" ++ shown_expr
	    Nothing -> return()
	let ev = substituteLibraryBuiltins uio ufn mm expr
	uv <- if tc then do
	    aside verbose "-- About to check types. --"
	    let UseLowerIOType lio = uio
	    let UseStrictMonad sm = mm
	    let type_to_check sp = if lio then notT sp (iOT sp noT "")
				     else if sm then iONT uio sp (notT sp noT)
				     else iONT uio sp noT
	    ev' <- case ev of
		TO (Type k sp NoT, _) coop -> return $ TO (type_to_check sp, no_span) coop
		TO (Type k _ (ApplicationT (Type _ _ (NameT "ION" _)) _), _) coop -> return ev
		TO (Type k _ (ApplicationT (Type _ _ NotT) (Type _ _ (ApplicationT (Type _ _ (NameT "IO" _)) _))), _) coop -> return ev
		otherwise -> die ("Expression is not an action:\n" ++ tshow TopShowSpan tt False ev)
	    XExpr uv <- unify verbose_unify tk uc ulimit mm uio ufn
				(types_of_native_builtins uio ufn mm) (XExpr ev')
	    liftIO $ hPutStrLn (maybe nonstandard_output id verbose) "--- Types checked. ---"
	    aside verbose $ strictList_ $ "to " ++ tshow NoShowSpan True tk uv
	    return uv
	  else do
	    aside verbose "-- Skipping, not checking types. --"
	    return ev
	liftIO $ cotester'' outdest uv

    cotester'' :: OutFileProxy -> {- Semval -} Coterm -> IO ()
    cotester'' outdest expr = do
	    let OutFileProxy open close reset message = outdest
	    case verbose of
		Just h -> do
		    hPutStrLn h "The unsubstituted expression:"
		    hPutStrLn h $ tshow NoShowSpan False False expr
		    hPutStrLn h $ show' expr
		Nothing -> return ()
	    runtime_output_ref <- newIORef Nothing
	    pv <- lower_coterm expr (create_stack (nativeBuiltins uio ufn mm runtime_output_ref))
	    let nv = (\e -> Cut_ "run3ning the program" run_term e) pv
	    case verbose of
		Just h -> do
		    hPutStrLn h "The original expression:"
		    hPutStrLn h $ show nv
		    hPutStrLn h "--- Starting evaluation. ---"
		Nothing -> return ()
	    runtime_output <- open
	    writeIORef runtime_output_ref (Just runtime_output)
	    evaluate nv
	    writeIORef runtime_output_ref Nothing
	    close runtime_output
	    message
	    return ()

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- Run_coterm has type lowered from IOV t.
-- This coterm runs the corresponding term, specifically discarding any
-- result value returned.
run_coterm :: Coterm_
run_coterm = Abstract_ "@run"
		(Cut_ "run4ning the program"
		    (Bind_ "the complete program" "the program body" "exit at the program end" (Variable_ 0) (Selector_ 2 1 (Use_ (Exit_ "run_coterm"))))
		    (Expose_ (Case_ []) (base_expose_id, "the complete program") Nothing)
		)
		(Just 1)

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- Run_term has type lowered from ION t.
run_term :: Term_
-- run_term = Plan_ (Expose_ (Case_ []))
run_term = Plan_ run_coterm

data OutFileProxy = OutFileProxy (IO Handle) (Handle -> IO ()) (IO ()) (IO ())
stdout_proxy verbose = OutFileProxy (return stdout) (const $ return ()) (return ()) (if verbose then putStrLn "--- Evaluation is stopped. ---" else return ())

run_file :: FilePath -> IO ()
run_file source_file_name = do
    program <- readUTF8File source_file_name
    run_string (default_parameters source_file_name) source_file_name (stdout_proxy False) program

test :: Parameters -> IO ()
test params = do
    let m_source_name = source_file params
    let m_output_name = output_file params
    let version = Parameters.version params
    let verbose = is_verbose params
    when version $ hPutStrLn stderr "Ambidexter 0.5"
    case m_source_name of
	Just source_name -> do
	    program <- readUTF8File source_name
	    let proxy = maybe 
		    (stdout_proxy verbose)
		    (\ofn -> OutFileProxy (openFile ofn WriteMode) hClose (remove ofn) (return ()))
		    m_output_name
	    run_string params "" proxy program
	Nothing -> return ()
  where
    remove fn = do
      exists <- doesFileExist fn
      if exists then removeFile fn
	        else return ()

main = do
    args <- getArgs
    m_params <- analyze_args args
    maybe (return ()) test m_params

