{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Parse (expression, parse, Parser) where

import Monad
import Maybe
import Char hiding (isAlpha, isAlphaNum)
import qualified Char
import Data.List (intersect)

import qualified Text.ParserCombinators.Parsec as Parsec
import Text.ParserCombinators.Parsec (Parser, (<|>), try, parse, eof, oneOf, noneOf)

import Position
import Kind
import Type
import AST
import Basic

require :: MonadPlus m => Maybe t -> m t
require Nothing	=  mzero
require (Just v)	=  return v

char = Parsec.char
(+++) a b = (Parsec.try a) <|> b
many = Parsec.many
many1 = Parsec.many1
sat = Parsec.satisfy
may_parse parser = fmap Just parser <|> return Nothing

-- === -- === -- === -- === -- === -- === -- === -- === -- === -- === -- === --
-- Parsers

expression :: UseFunctionTypes -> Bool -> Parser (Either Term Coterm)
expression (UseLowerFunctionTypes lfn) coerce = expression'
  where
    -- expression ::= w* (parenthesized_term | bracketed_coterm) w*
    expression' = do
	    start <- get_pos
	    whites
	    v <- fmap Left parenthesized_term <|> fmap Right bracketed_coterm
	    whites
	    v' <- case v of
			    Left a -> fmap Left $ spanify' start a
			    Right b -> fmap Right $ spanify' start b
	    eof
	    return v'

    -- parenthesized_term ::= '(' w* term w* (| annotation w*) ')'
    parenthesized_term = do
			    start <- get_pos
			    char '('
			    whites
			    v <- term
			    whites
			    mt <- (do
				    t <- annotation
				    whites
				    return $ Just t) <|> return Nothing
			    char ')'
			    if False then
				-- for reorient only
				spanify' start $ (TM (maybe noT id mt,no_span) . AnnotatedTerm False) v
			     else
				spanify' start $ (maybe id (\tp -> TM (tp,no_span) . AnnotatedTerm False) mt) v

    --      ,        \           azAZ09"({                    &              [
    term = list						-- ,
	    <|> abstraction					-- azAZ|-@[(
	    <|> product_ctor				-- &
	    <|> fmap (\br@(TO (tp,sp) c) -> TM (noT,sp) (PlanTerm br)) bracketed_coterm	-- [
	    <|> secondary True					-- azAZ09"({\@
    -- secondary ::= application
    secondary allow_empty = lambda <|> generalize <|> application allow_empty

    {- Certain primaries are not allowed to succeed one another without intervening whitespace,
       i.e. id or integer followed by id or integer
       This is addressed using lookahead in the processing of id and integer.
     -}

    -- primary begins with azAZ09"({
    look_secondary = look_satisfy (\ch -> isAlphaNum ch || ch `elem` "\"({_\\@")
    primary allow_empty = id_expr <|> integer <|> string <|>  parenthesized_term <|> binding <|> (if allow_empty then true_ctor else mzero)

    -- Lists of terms require whitespace between all successive terms,
    -- even when there are delimiters involved, e.g. 1(2) is invalid.
    -- list ::= ',' (w* secondary w* ',')*
    list :: Parser Term
    list = do 
	      start <- get_pos
	      char ','
	      d <- (do 
		  look_secondary <|> look_satisfy (== ',')
		  _ <- whites
		  s <- secondary True
		  _ <- whites
		  TM _ (ListTerm l) <- list
		  return $ TM noTP $ ListTerm (s : l))
	       <|> (return $ TM noTP $ ListTerm [])
	      spanify' start d

    -- Parse juxtaposed terms as function application.
    -- application ::= primary ( w+ secondary | )
    -- Without that, error messages may be way off.
    application allow_empty = do
		    start <- get_pos
		    t <- primary allow_empty
		    non_empty <- pos_is_changed start
		    (do
			expr <- (do 
				    guard non_empty
				    look_secondary
				    ws <- whites
				    e <- secondary False
				    return $ Just e)
				<|>
				(return Nothing)
			return $ maybe t (\e -> spanify_term start e TM noT $ ApplicationTerm t e) expr
			  )

    -- lambda ::= '\' w* identifier w* '->' secondary
    lambda = do
		start <- get_pos
		char '\\'
		whites
		pat <- pattern
		whites
		char '-'
		char '>'
		whites
		expr <- secondary True
		return $ spanify_term start expr TM noT $ LambdaTerm pat expr

    -- generalize ::= '@' w* identifier w* ':>' secondary
    generalize = do
		start <- get_pos
		char '@'
		whites
		id <- identifier
		whites
		char ':'
		char '>'
		whites
		expr <- secondary True
		spanify start TM (Type NoKind no_span NoT) (GenericTypeTerm (Type NoKind no_span NoT, no_span) id expr)

    -- binding ::= '{' (w* secondary w* ("=:" w* identifier w* |) ';')* w* secondary w* '}'
    -- binding ::= '{' w* secondary w* (("=:" w* identifier w* |) ';' w* secondary w*)* '}'
    binding = do
		start <- get_pos
		char '{'
		val <- binding_guts
		char '}'
		spanify' start val
      where
	binding_guts = do
		whites
		term <- secondary True
		whites
		more term <|> return term
	finish_definition = do
		mpat <- may_parse (do 
			char '='
			char ':'
			whites
			start <- get_pos
			pat <- pattern
			span <- span_to_here start
			whites
			return (pat, span)
			  )
		char ';'
		return mpat
	more lval = do
		mpat <- finish_definition
		whites
		rval <- binding_guts
		whites
		let spn = pos_span (start_pos_of lval) (end_pos_of rval)
		return $ add_binding spn lval mpat rval
	add_binding spn lterm mpat rval = TM (noT, spn) $ maybe
		    (SeqTerm lterm rval)
		    (\(Pattern (t,_) pat, span) -> BindTerm lterm (TM noTP $ LambdaTerm (Pattern (t, span) $ pat) rval))
		    mpat

    -- cobinding ::= '{' (w* cosecondary w* ("=:" w* identifier w* |) ';')* w* cosecondary w* '}'
    -- cobinding ::= '{' w* cosecondary w* (("=:" w* identifier w* |) ';' w* cosecondary w*)* '}'
    cobinding = do
		start <- get_pos
		char '{'
		val <- cobinding_guts
		char '}'
		spanify' start val
      where
	cobinding_guts = do
		whites
		term <- cosecondary True
		whites
		more term <|> return term
	finish_definition = do
		mpat <- may_parse (do 
			char '='
			char ':'
			whites
			start <- get_pos
			pat <- copattern
			span <- span_to_here start
			whites
			return (pat, span)
			  )
		char ';'
		return mpat
	more lval = do
		mpat <- finish_definition
		whites
		rval <- cobinding_guts
		whites
		let spn = pos_span (start_pos_of lval) (end_pos_of rval)
		return $ add_binding spn lval mpat rval
	add_binding spn lterm mpat rval = TO (noT, spn) $ maybe
		    (CoseqTerm lterm rval)
		    (\(Copattern (t,_) pat, span) -> CobindTerm lterm (TO noTP $ ColambdaTerm (Copattern (t, span) pat) rval))
		    mpat

    -- product_ctor ::= '&' (w* secondary w* '&')*
    product_ctor = do
		    start <- get_pos
		    char '&'
		    d <- (do 
			look_secondary <|> look_satisfy (== '&')
			_ <- whites
			s <- secondary True
			_ <- whites
			TM _ (ProductTerm l) <- product_ctor
			return $ TM noTP $ ProductTerm (s : l))
		     <|> (return $ TM noTP $ ProductTerm [])
		    spanify' start d

    -- true_ctor ::= 
    true_ctor = do
		    start <- get_pos
		    spanify start TM noT $ VoidTerm

    -- abstraction ::= copattern w* '<+' w* secondary w* '$' w* cosecondary
    abstraction = do
		    look_abstraction ["<+"]
		    copat <- copattern
		    whites
		    char '<'
		    char '+'
		    whites
		    exp <- secondary True
		    whites
		    char '$'
		    whites
		    coexp <- cosecondary True
		    spanify (start_pos_of exp) TM noT $ AbstractTerm copat exp coexp

    -- bracketed_coterm ::= '[' w* coterm w* (| annotation w* ) ']'
    bracketed_coterm :: Parser Coterm
    bracketed_coterm = do
			    start <- get_pos
			    char '['
			    whites
			    v <- coterm
			    whites
			    mt <- (do
				    t <- annotation
				    whites
				    return $ Just t) <|> return Nothing
			    char ']'
			    if False then
				-- for reorient only
				spanify' start $ (TO (maybe noT id mt,no_span) . CoannotatedTerm False) v
			     else
				spanify' start $ (maybe id (\tp -> TO (tp,no_span) . CoannotatedTerm False) mt) v
    -- coterm ::= abstract_coterm | sum_ctor | colist | colambda | package
    coterm :: Parser Coterm
    coterm =
		abstract_coterm	-- &@azAZ-[(
	    <|> sum_ctor		-- |
	    <|> colist		-- ,
	    <|> fmap (\pa@(TM (tp,sp) t) -> TO (noT,sp) (UseTerm pa))  parenthesized_term	-- (
	    <|> cosecondary True		-- azAZ09"[{\@
    -- cosecondary ::= coapplication
    cosecondary:: Bool -> Parser Coterm
    cosecondary allow_empty = colambda <|> package <|> coapplication allow_empty
    -- Parse juxtaposed coterms as cofunction application.
    -- coapplication ::= ( coapplication w* | ) coprimary
    coapplication :: Bool -> Parser Coterm
    coapplication allow_empty = do
		    start <- get_pos
		    t <- coprimary allow_empty
		    non_empty <- pos_is_changed start
		    (do
			expr <- (do 
				    guard non_empty
				    look_cosecondary
				    ws <- whites
				    e <- cosecondary False
				    return $ Just e)
				<|>
				(return Nothing)
			return $ maybe t (\e -> spanify_term start e TO noT $ CoapplicationTerm t e) expr
			  )

    -- starts with azAZ                  [                             09                           "              {
    -- returns an extra flag to indicate whether the primary needs to be followed by whitespace before another primary
    coprimary :: Bool -> Parser Coterm
    look_cosecondary = look_satisfy (\ch -> isAlphaNum ch || ch `elem` "\"[{_\\@")
    coprimary allow_empty = id_coexpr <|> bracketed_coterm <|> addgo integer <|> addgo string <|> cobinding <|> (if allow_empty then false_ctor else mzero)
	where
	    addgo = fmap (\x@(TM (_,sp) _) -> TO (noT,sp) (UseTerm x))

    -- sum_ctor ::= '|' (w* cosecondary w* '|')*
    sum_ctor = do
		    start <- get_pos
		    char '|'
		    d <- (do
			look_cosecondary <|> look_satisfy (== '|')
			_ <- whites
			s <- cosecondary True
			_ <- whites
			TO _ (SumTerm l) <- sum_ctor
			return $ TO noTP $ SumTerm (s : l))
		     <|> (return $ TO noTP $ SumTerm [])
		    spanify' start d

    -- false_ctor ::=
    false_ctor = do
		    start <- get_pos
		    spanify start TO noT $ UnreachedTerm

    look_abstraction hyphens = do
		    input <- Parsec.getInput
		    let arrow_from_nesting n s = case s of
			    (x:xs) -> case x of
				'(' -> arrow_from_nesting (n+1) xs
				'[' -> arrow_from_nesting (n+1) xs
				')' -> if n == 0 then False else arrow_from_nesting (n-1) xs
				']' -> if n == 0 then False else arrow_from_nesting (n-1) xs
				'"' -> False
				otherwise -> if n == 0 then case found of
					    (_:_) -> True
					    _ -> arrow_from_nesting n xs
					else arrow_from_nesting n xs
				    where
					found = catMaybes (map (`initial_match` s) hyphens)
			    otherwise -> False
		    if arrow_from_nesting 0 input then return () else mzero

    -- abstract_coterm ::= pattern w* '->' w* secondary w* '$' w* cosecondary
    abstract_coterm = do
		    look_abstraction ["->"]
		    start <- get_pos
		    pat <- pattern
		    whites
		    char '-'
		    char '>'
		    whites
		    exp <- secondary True
		    whites
		    char '$'
		    whites
		    coexp <- cosecondary True
		    spanify start TO noT $ CoabstractTerm pat exp coexp

    -- colambda ::= '\' w* copattern w* '<+' cosecondary
    colambda = do
		start <- get_pos
		char '\\'
		whites
		look_abstraction ["<+"]
		copat <- copattern
		whites
		char '<'
		char '+'
		whites
		expr <- cosecondary True
		return $ spanify_term start expr TO noT $ ColambdaTerm copat expr

    -- package ::= '@' w* identifier w* '<:' cosecondary
    package = do
		start <- get_pos
		char '@'
		whites
		id <- identifier
		whites
		char '<'
		char ':'
		whites
		expr <- cosecondary True
		spanify start TO (Type NoKind no_span NoT) $ AbstractTypeTerm (Type NoKind no_span NoT, no_span) id expr

    -- colist ::= ',' (w* cosecondary w* ',')*
    colist :: Parser Coterm
    colist = do
	    start <- get_pos
	    char ','
	    d <- (do
		look_cosecondary <|> look_satisfy (== ',')
		_ <- whites
		s <- cosecondary True
		_ <- whites
		TO _ (ColistTerm l) <- colist
		return $ TO noTP $ ColistTerm (s : l))
	     <|> (return $ TO noTP $ ColistTerm [])
	    spanify' start d

    -- pattern ::= pattern_primary | product_pattern | specific_pattern | true_pattern
    pattern :: Parser Pattern
    pattern = product_pattern
		    <|> specific_pattern
		    <|> pattern_primary

    -- product_pattern ::= '&' (w* identifier w* '&')*
    product_pattern :: Parser Pattern
    product_pattern = do
		    start <- get_pos
		    char '&'
		    d <- (do
			    look_satisfy (\ch -> ch == '&' || isAlpha ch || ch == '(' || ch == '[' || ch == '_')
			    whites
			    mpat <- pattern_primary_or_placeholder
			    whites
			    p@(Pattern _ (AndPat pats)) <- product_pattern
			    return $ Pattern noTP $ AndPat (mpat:pats))
		     <|> (return $ Pattern noTP $ AndPat [])
		    spanify' start d

    -- true_pattern ::=
    true_pattern :: Parser Pattern
    true_pattern = do
		    start <- get_pos
		    input <- Parsec.getInput
		    look_match ")" <|> look_match "->" <|> look_match "&"
		    spanify start Pattern noT VoidPat

    -- specific_pattern ::= '@' w* (typex w* ':>' w*) pattern
    specific_pattern = do
		    start <- get_pos
		    char '@'
		    tp@(Type k tsp _) <- try (do
			    look_satisfy (\ch -> ch == '~' || ch == '\\' || isAlphaNum ch || ch == '(' || ch == '[') <|> look_match unicode_not
			    whites
			    tp <- typex
			    whites
			    char ':'
			    char '>'
			    whites
			    return tp
			) <|> (do
			    tstart <- get_pos
			    sp <- span_to_here tstart
			    return (Type NoKind sp NoT))
		    pat <- pattern
		    spanify start Pattern noT $ SpecificPat (TypeExpression (tp,tsp)) pat

    -- not_pattern ::= '[' copattern ']'
    not_pattern = do
		    start <- get_pos
		    char '['
		    whites
		    copat <- copattern
		    whites
		    char ']'
		    spanify start Pattern noT $ NotPat copat

    -- pattern_primary_or_placeholder ::= identifier | not_pattern | '(' pattern ')' | '_'
    pattern_primary_or_placeholder :: Parser (Maybe Pattern)
    pattern_primary_or_placeholder = do
		start <- get_pos
		p <- fmap (fmap (Pattern noTP . NamePat)) identifier_or_placeholder
		    <|> fmap Just true_pattern
		    <|> fmap Just not_pattern
		    <|> do
			    char '('
			    whites
			    pat <- pattern
			    whites
			    char ')'
			    return $ Just pat
		case p of
		    Nothing -> return Nothing
		    Just p' -> fmap Just (spanify' start p')
    pattern_primary = do
		    maybe_pat <- pattern_primary_or_placeholder
		    case maybe_pat of
			Nothing -> fail "_ is not a pattern."
			Just pat -> return pat

    -- copattern ::= copattern_primary | sum_copattern | package_copattern | false_copattern
    copattern = sum_copattern
		    <|> package_copattern
		    <|> copattern_primary

    -- sum_copattern ::= '|' (w* identifier w* '|')*
    sum_copattern = do
		    start <- get_pos
		    char '|'
		    d <- (do
			    look_satisfy (\c -> c == '|' || isAlpha c || c == '(' || c == '[' || c == '_')
			    whites
			    mpat <- copattern_primary_or_placeholder
			    whites
			    p@(Copattern _ (OrCopat pats)) <- sum_copattern
			    return $ Copattern noTP $ OrCopat (mpat:pats))
		     <|> (return $ Copattern noTP $ OrCopat [])
		    spanify' start d

    -- false_copattern ::=
    false_copattern :: Parser Copattern
    false_copattern = do
		    start <- get_pos
		    look_match "]" <|> look_match "<+" <|> look_match "|"
		    spanify start Copattern noT UnreachedCopat

    -- not_copattern ::= '(' pattern ')'
    not_copattern = do
		    start <- get_pos
		    char '('
		    whites
		    pat <- pattern
		    whites
		    char ')'
		    spanify start Copattern noT $ NotCopat pat

    -- package_copattern ::= '@' w* ( typex w* ':>' w* | ) copattern
    package_copattern = do
		    start <- get_pos
		    char '@'
		    tp@(Type k tsp _) <- try (do
			    look_satisfy (\ch -> ch == '~' || ch == '\\' || isAlphaNum ch || ch == '(' || ch == '[') <|> look_match unicode_not
			    whites
			    tp <- typex
			    whites
			    char '<'
			    char ':'
			    whites
			    return tp
			) <|> return (Type NoKind no_span NoT)
		    copat <- copattern
		    spanify start Copattern noT $ PackageCopat (TypeExpression (tp, tsp)) copat

    -- copattern_primary_or_placeholder ::= identifier | not_copattern | '[' copattern ']' | '_'
    copattern_primary_or_placeholder = do
		start <- get_pos
		f <- fmap (fmap (Copattern noTP . NameCopat)) identifier_or_placeholder
		    <|> fmap Just not_copattern
		    <|> fmap Just false_copattern
		    <|> do
			    char '['
			    whites
			    copat <- copattern
			    whites
			    char ']'
			    return $ Just copat
		case f of
		    Nothing -> return Nothing
		    Just f' -> fmap Just (spanify' start f')
    copattern_primary = do
		    maybe_copat <- copattern_primary_or_placeholder
		    case maybe_copat of
			Nothing -> fail "_ is not a pattern."
			Just copat -> return copat

    -- type expression
    -- starts with ~azAZ(
    typex :: Parser Type
    typex = do
		    input <- Parsec.getInput
		    let starts = isJust . (`initial_match` input)
		    if starts unicode_all then quantified unicode_all ForallT
		     else if starts "all" then quantified "all" ForallT
		     else if starts unicode_some then quantified unicode_some ExistsT
		     else if starts "some" then quantified "some" ExistsT
		     else if starts "\\" then quantified "\\" LambdaT
		     else type_quaternary
	where
	    quantified str tc = do
		    start <- get_pos
		    Parsec.string str
		    if isAlphaNum (head (reverse str)) then white else return ' '
		    whites
		    id <- identifier
		    whites
		    char '.'
		    whites
		    tp <- typex
		    sp <- span_to_here start
		    return $ Type NoKind sp (tc (Type NoKind no_span NoT) id tp)

    type_quaternary :: Parser Type
    type_quaternary = do
	    t1@(Type _ sp1 _) <- type_tertiary
	    finish_function t1 sp1 <|> return t1
	where
	    finish_function t1 sp1 = do
		    look_matches (implies ++ not_implied_by)
		    whites
		    co <- (implies_token >> return False)
			    <|> (not_implied_by_token >> return True)
		    whites
		    t2@(Type _ sp2 _) <- type_quaternary
		    sp <- span_to_here (start_pos_of t1)
		    let tc = if not co then (if lfn then lower_function_type sp sp2 else FunctionT)
			               else (if lfn then lower_cofunction_type sp1 else CofunctionT)
		    return $ Type Proposition sp (tc t1 t2)
    -- starts with  azAZ       (       (      (        [          (
    type_primary :: Parser Type
    type_primary = builtin <|> type_primary_starts_with_parenthesis <|> colist
	where
	    builtin = do
		    start <- get_pos
		    id_or_pl <- identifier_or_placeholder
		    sp <- span_to_here start
		    let mtp = case id_or_pl of
			    Nothing -> Just (set_pos sp noT)
			    Just id -> Just $ Type NoKind sp (NameT id "")
		    require mtp
	    type_primary_starts_with_parenthesis = do
		    start <- get_pos
		    char '('
		    list start <|> (whites >> (and start <|> or start <|> parenthesized_type))
	    and start = do
		    -- char '('
		    look_matches ["&", unicode_and]
		    whites
		    and_token
		    whites
		    tps <- many another_and_type
		    char ')'
		    sp <- span_to_here start
		    return $ prop sp $ AndT PlainOrigin tps
	    another_and_type = do
		    t <- typex
		    whites
		    and_token
		    whites
		    return t
	    or start = do
		    -- char '('
		    look_matches ["|", unicode_or]
		    whites
		    or_token
		    whites
		    tps <- many another_or_type
		    char ')'
		    sp <- span_to_here start
		    return $ prop sp $ OrT tps
	    another_or_type = do
		    t <- typex
		    whites
		    or_token
		    whites
		    return t
	    list start = do
		    -- char '('
		    char ','
		    whites
		    tp <- typex
		    whites
		    char ','
		    char ')'
		    sp <- span_to_here start
		    return $ prop sp $ ApplicationT (Type (Predicate (Kind Proposition) (Kind Proposition)) sp (NameT "List" "")) tp
	    colist = do
		    start <- get_pos
		    char '['
		    char ','
		    whites
		    tp <- typex
		    whites
		    char ','
		    char ']'
		    sp <- span_to_here start
		    return $ prop sp $ ApplicationT (Type (Predicate (Kind Proposition) (Kind Proposition)) sp (NameT "Colist" "")) tp
	    parenthesized_type = do
		    -- char '('
		    -- whites
		    tp <- typex
		    whites
		    char ')'
		    return tp
	    prop = Type Proposition
    -- starts with   ~                    I                   I              azAZ (
    type_tertiary :: Parser Type
    type_tertiary = do
		    input <- Parsec.getInput
		    let starts = isJust . (`initial_match` input)
		    if starts unicode_not then sec unicode_not (\sp -> ApplicationT (ctorT_ sp NotT))
		     else if starts "~" then sec "~" (\sp -> ApplicationT (ctorT_ sp NotT))
		     else type_secondary
	where
	    sec str tc = do
		    start <- get_pos
		    Parsec.string str
		    if isAlphaNum (head (reverse str)) then white else return ' '
		    whites
		    tp <- type_tertiary
		    sp <- span_to_here start
		    return $ Type Proposition sp (tc sp tp)
    -- starts with   ~                    I                   I              azAZ (
    type_secondary :: Parser Type
    type_secondary = finish_secondary return
	where
	    -- Maker takes the next primary and makes the next secondary.
	    finish_secondary maker = do
		    input <- Parsec.getInput
		    let is_id = case input of
			    (ch:_) -> isAlphaNum ch
			    _ -> False
		    pri <- is_id `seq` type_primary
		    sec <- maker pri
		    (do
			if is_id then white else return ' '
			whites
			finish_secondary (combiner sec))
		      +++ return sec
	    combiner constructor arg = do
		sp <- span_to_here (start_pos_of constructor)
		return $ Type NoKind sp (ApplicationT constructor arg)

    -- type annotation
    annotation :: Parser Type
    annotation = do
		    char ':'
		    whites
		    typex

-- identifier ::= [a-zA-Z_]+
-- An identifier consisting of a single underscore is a placeholder.
identifier_or_placeholder = do
	    name <- many1 (sat (\ch -> isAlpha ch || ch == '_'))
	    input <- Parsec.getInput
	    case input of
		(a:as) | isAlpha a || a == '_' -> error "many1 was not greedy enough"
		(a:as) | isDigit a -> fail "invalid identifier"
		otherwise -> if name == "_" then return Nothing
				else return $ Just name
identifier = do
	    maybe_name <- identifier_or_placeholder
	    case maybe_name of
		Nothing -> fail "_ is not an identifier."
		Just name -> return name

-- id_expr ::= identifier
id_expr :: Parser Term
id_expr = do
	    start <- get_pos
	    id <- fmap (TM noTP . VariableTerm) identifier
	    spanify' start id
-- id_coexpr ::= identifer
id_coexpr :: Parser Coterm
id_coexpr = do
	    start <- get_pos
	    id <- fmap (TO noTP . CovariableTerm) identifier
	    spanify' start id

-- integer ::= [0-9]+
integer = do
	    start <- get_pos
	    n <- many1 (sat isDigit)
	    input <- Parsec.getInput
	    case input of
		(a:as) | isDigit a -> error "many1 was not greedy enough"
		(a:as) | isAlpha a || a == '_' -> fail "invalid number"
		otherwise -> spanify start TM noT $ IntegerTerm (read n)

-- string ::= '"' [^"\\\n]* '"'
-- !! Need to decide what's the point of disallowing backslash.  Just to avoid confusion?
string = do
	    start <- get_pos
	    char '"'
	    str <- many (sat (\ch -> ch /= '"' && ch /= '\n' && ch /= '\\'))
	    char '"'
	    spanify start TM noT $ StringTerm str

-- w ::= [ \n\t\r\f]
-- !! Need to be explicit here, and not just rely on Haskell's definition of whitespace.
white = sat isSpace

get_pos = do
		inp <- Parsec.getInput
		pos <- Parsec.getPosition
		let sl = Parsec.sourceLine pos
		let sc = Parsec.sourceColumn pos - 1
		return $! Position sl sc inp

noTP = (noT,no_span)

spanify :: Position -> ((Type, Span) -> s -> c (Type, Span)) -> Type -> s -> Parser (c (Type, Span))
spanify start ctor typ val = do
	sp <- span_to_here start
	return $ ctor (typ,sp) val
spanify_term :: (Positionable p) => Position -> p -> ((Type, Span) -> s -> c (Type, Span)) -> Type -> s -> c (Type, Span)
spanify_term start end_term ctor typ val = ctor (typ,pos_span start end) val
    where
	(Span _ end _) = get_span end_term

spanify' :: (Positionable t) => Position -> t -> Parser t
spanify' start val = do
	sp <- span_to_here start
	return $ set_pos sp val

span_to_here start_pos@(Position l c str) = do
		end_pos <- get_pos
		return $ pos_span start_pos end_pos

pos_is_changed :: Position -> Parser Bool
pos_is_changed (Position original_line original_column _) = do
		(Position line column _) <- get_pos
		return $ line /= original_line || column /= original_column

whites :: Parser String
whites = many whiteOrComment
  where
    whiteOrComment = white <|> (comment >> return ' ')
    comment = do
	char '#'
	many (noneOf "\n\r")
	(oneOf "\n\r") <|> (eof >> return ' ')
	return ()

implies = ["->", unicode_implies]
implies_token = do
	look_matches implies
	Parsec.string "->" <|> Parsec.string unicode_implies
not_implied_by = ["<+", unicode_not_implied_by]
not_implied_by_token = do
	look_matches not_implied_by
	Parsec.string "<+" <|> Parsec.string unicode_not_implied_by
and_token = Parsec.string "&" <|> Parsec.string unicode_and
or_token = Parsec.string "|" <|> Parsec.string unicode_or
unicode_implies :: String
unicode_implies = [toEnum 0x2192] -- right arrow
unicode_not_implied_by :: String
unicode_not_implied_by = [toEnum 0x219A] -- left arrow with stroke
unicode_and :: String
unicode_and = [toEnum 0x2227]
unicode_or :: String
unicode_or = [toEnum 0x2228]
unicode_not :: String
unicode_not = [toEnum 0xAC]
unicode_all :: String
unicode_all = [toEnum 0x2200]
unicode_some :: String
unicode_some = [toEnum 0x2203]

-- We're parsing bytes as they come from I/O, interpreted as UTF-8.
-- Char.isAlpha assumes iso8859-1.
isAlpha ch = Char.isAlpha ch && fromEnum ch < 128
isAlphaNum ch = Char.isAlphaNum ch && fromEnum ch < 128

-- Require that the lookahead (current input) start with a certain string,
-- after optional whitespace.
look_match :: String -> Parser ()
look_match s = do
	input <- Parsec.getInput
	let next = look_beyond_whites input
	if take (length s) next == s then return () else mzero
-- Require one of the strings.
look_matches :: [String] -> Parser ()
look_matches ss = case ss of
	[] -> mzero
	(s:ss) -> look_match s <|> look_matches ss

look_satisfy :: (Char -> Bool) -> Parser ()
look_satisfy pred = do
	input <- Parsec.getInput
	let next = look_beyond_whites input
	let satisfied = case next of
		[] -> False -- pred Nothing
		(ch:chs) -> pred ch
	if satisfied then return () else mzero

look_beyond_whites :: String -> String
look_beyond_whites s = case s of
	('#' : xs) -> look_beyond_whites (after_comment xs)
	(x : xs) | isSpace x -> look_beyond_whites xs
	xs -> xs
  where
    after_comment s = case s of
	(x:xs) -> if x `elem` "\n\r" then xs else after_comment xs
	[] -> []

initial_match :: String -> String -> Maybe String
initial_match test input = if test == input0 then Just input1 else Nothing
    where
        (input0, input1) = splitAt (length test) input
