{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module Type (
	Knd(NoKind, Proposition, Predicate, VarKind), Kind(Kind),
	Tp(
		VarT, AndT, OrT, NotT, NoT,
		FunctionT, YVT, YNT,
		CofunctionT, NameT, ExistsT, ForallT,
		ApplicationT, LambdaT, UnsatisfiedT),
	Type(Type),
	TypeInContext(TypeInContext), TypeContext,
	FunctionOrigin(FunctionOrigin, CofunctionOrigin, PlainOrigin),
	noT, notT, nameT, ctorT, ctorT_, functionT, cofunctionT, forallT, existsT,
	stringT, booleanT, integerT, andT, orT, iOVT, iONT, applicationT, applicationT_,
	iOT, iOT_V, iOT_N,
	trueT, falseT, listT, colistT, prop,
	lower_function_type, lower_cofunction_type,
	uprop_kind, unoT, uprop, kprop,
	uctorT, uctorT_,
	uIOVT, uIONT, u_mT, u_mVT, u_mNT, ulistT, ucolistT,
	UTRef, UType(UType),
	cc,
	UseIOType(UseLowerIOType),
	UseFunctionTypes(UseLowerFunctionTypes),
	MonadModel(UseStrictMonad),
	MatchGet,
	m_any, m_application, m_function, m_cofunction, m_and_2, m_app_not, m_not, m_prop_name, m_ctor_name,
	m_monad_v, m_monad_n, m_prop, maybe_do
	) where
import Monad
import List
import Maybe
import Data.Map
import Data.IORef
import Control.Monad.State

import Basic
import Position

data Kind = Kind (Knd Kind)
data Knd ukref
	= NoKind
	| Proposition
	| Predicate ukref ukref
	| VarKind TypeId UTRef	-- kind vars introduced by unification
	deriving (Eq)

-- cycle check
cc :: (MonadIO m) => String -> UTRef -> m ()
cc msg tvar = safe_more_tvars True tvar tvar tvar
    where
	safe_more_tvars pause stopper to_tvar from_tvar = do
	    this_var <- liftIO $ readIORef from_tvar
	    case this_var of
		UType k sp (VarT ids tvarref) -> if tvarref == to_tvar then return ()
				    else do
					stopper' <- if pause then return stopper
					    else do
						    UType k sp (VarT ids next) <- liftIO $ readIORef stopper
						    if tvarref == next then die ("internal error: entry into cycle for " ++ show ids ++ " at " ++ msg)
							else return ()
						    return next
					safe_more_tvars (not pause) stopper' to_tvar tvarref
		otherwise -> return () -- ignore because sometimes we need to guard against cycles in contexts where the type is not always VarT.    die ("unified a non-type-variable (" ++ msg ++ ")")


data FunctionOrigin = FunctionOrigin | CofunctionOrigin | PlainOrigin
data Type = Type (Knd Kind) Span (Tp Type)
data Tp typ = VarT TypeId UTRef -- type vars introduced by unification
	| NameT VarName String	-- scoped abstract type name, or quantified type parameter
	| ApplicationT typ typ | LambdaT typ VarName typ
	| NoT
	| AndT FunctionOrigin [typ] | OrT [typ]
	| NotT
	| FunctionT typ typ
	| YVT typ | YNT typ
	| CofunctionT typ typ
	| ExistsT typ VarName typ | ForallT typ VarName typ
	| UnsatisfiedT (StateT [TypeId] IO (Maybe (TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String))) String

-- The first member ("kind") of LambdaT, ExistsT, and ForallT are typ rather than k so that
-- they can hold a VarT which is a proxy for the named type and which can be unified with
-- other named types.

-- YVT Type is the type of a function which takes itself as its first argument.
-- Note that it is not the type of Y (or fix), and is not for source-level use.
-- That would be an infinite type if we didn't use YVT.  
--    YVT t == (YVT t) -> t         YNT t == (YNT t) <+ t
-- See example in Pierce, "Recursive Values from Recursive Types".
-- The YXT_ types are similar, but work with lowered types.

newtype UseIOType = UseLowerIOType Bool
newtype UseFunctionTypes = UseLowerFunctionTypes Bool
newtype MonadModel = UseStrictMonad Bool

type UTRef = IORef UType
data UType = UType (Knd UTRef) Span (Tp UTRef)

type TypeContext = Map VarName (Way, TypeInContext)
data TypeInContext = TypeInContext UTRef TypeContext

prop = Type Proposition
uprop = UType Proposition
kprop = Kind Proposition
noT = prop no_span NoT
notT sp x = prop sp $ ApplicationT (ctorT_ sp NotT) x
nameT sp x = prop sp (NameT x "")
ctorT_ sp x = Type (Predicate (Kind Proposition) (Kind Proposition)) sp x
ctorT sp x = ctorT_ sp (NameT x "")
functionT (UseLowerFunctionTypes lft) sp x y = if not lft then prop sp (FunctionT x y)
	else notT sp (prop sp (AndT FunctionOrigin [x, notT sp y]))
cofunctionT (UseLowerFunctionTypes lft) sp x y
    = if not lft then prop sp (CofunctionT x y)
	else prop sp (AndT CofunctionOrigin [notT sp x, y])
forallT sp k x y = prop sp (ForallT k x y)
existsT sp k x y = prop sp (ExistsT k x y)
stringT sp = nameT sp "String"
booleanT sp = nameT sp "Boolean"
integerT sp = nameT sp "Integer"
andT sp origin ts = prop sp (AndT origin ts)
orT sp ts = prop sp (OrT ts)
iOVT (UseLowerIOType lio) sp x = if lio
    then iOT_V sp x
    else prop sp $ ApplicationT (ctorT sp "IOV") x
iONT (UseLowerIOType lio) sp x = if lio
    then error "iONT with lowering types"
    else prop sp $ ApplicationT (ctorT sp "ION") x
iOT sp x orig = prop sp $ ApplicationT (ctorT_ sp (NameT "IO" orig)) x
iOT_V sp x = iOT sp x "V"
iOT_N (UseStrictMonad sm) sp x = iOT sp x (if sm then "S" else "N")
applicationT sp x = prop sp $ ApplicationT (ctorT_ sp NoT) x
applicationT_ sp = applicationT sp (prop sp NoT)
trueT sp = nameT sp "True"
falseT sp = nameT sp "False"
listT sp x = prop sp $ ApplicationT (ctorT sp "List") x
colistT sp x = prop sp $ ApplicationT (ctorT sp "Colist") x

lower_function_type span tspan s t = ApplicationT (ctorT_ span NotT) (andT span FunctionOrigin [s, notT tspan t])
lower_cofunction_type sspan s t
	= AndT CofunctionOrigin [notT sspan s, t]

uprop_kind :: (MonadIO m) => m UTRef
uprop_kind = liftIO $ newIORef (UType Proposition no_span NoT)
unoT :: Span -> StateT [TypeId] IO UTRef
unoT sp = do
	ref <- liftIO $ newIORef (UType Proposition sp NoT)
	(v:_) <- liftState get
	liftState (modify tail)
	liftIO $ writeIORef ref (UType Proposition sp (VarT v ref))
	cc "7" ref
	return ref
uIOVT uio@(UseLowerIOType lio) sp x = do
	ctor_tref <- uctorT_ sp (if lio then NameT "IO" "V" else NameT "IOV" "")
	u_mVT uio sp ctor_tref x
u_mVT (UseLowerIOType lio) sp m x = liftIO $ newIORef (uprop sp (ApplicationT m x))
uIONT uio@(UseLowerIOType lio) sp x = do
	ctor_tref <- uctorT_ sp (if lio then error "uIONT with lowering" else NameT "ION" "")
	u_mNT uio sp ctor_tref x
u_mNT (UseLowerIOType lio) sp m x = if lio
    then do
	notc_tref <- uctorT_ sp NotT
	not_tref <- u_mT sp notc_tref x
	u_mT sp m not_tref
    else do
	liftIO $ newIORef (uprop sp (ApplicationT m x))
uIOT sp x orig = do
	ctor_tref <- uctorT_ sp (NameT "IO" orig)
	u_mT sp ctor_tref x
u_mT sp m x = do
	liftIO $ newIORef (uprop sp (ApplicationT m x))
uctorT sp x = uctorT_ sp (NameT x "")
uctorT_ sp x = liftIO $ do
	kref1 <- uprop_kind
	kref2 <- uprop_kind
	newIORef (UType (Predicate kref1 kref2) sp x)
ulistT sp x = do
	ctor_tref <- uctorT sp "List"
	liftIO $ newIORef (uprop sp $ ApplicationT ctor_tref x)
ucolistT sp x = do
	ctor_tref <- uctorT sp "Colist"
	liftIO $ newIORef (uprop sp $ ApplicationT ctor_tref x)

-- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == -- == --

type MatchGet m = (UTRef -> m (Maybe [UTRef]))
type KindMatchGet m = (Knd UTRef -> m (Maybe [UTRef]))

maybe_do :: (Functor m, Monad m) => Maybe (m t) -> m (Maybe t)
maybe_do mf = maybe (return Nothing) (fmap Just) mf

m_any :: (Monad m) => MatchGet m
m_any tp_ref = return (Just ([tp_ref]))
m_application :: (Monad m, MonadIO m) => MatchGet m -> MatchGet m -> MatchGet m
m_application g1 g2 tp_ref = do
	tp <- liftIO (readIORef tp_ref)
	case tp of
 	    UType Proposition _ (ApplicationT n s) -> do
				mm1 <- g1 n
				mm2 <- g2 s
				return (do
					m1 <- mm1
					m2 <- mm2
					return (m1 ++ m2))
	    otherwise -> return Nothing
m_function :: (MonadIO m) => UseFunctionTypes -> MatchGet m -> MatchGet m -> MatchGet m
m_function (UseLowerFunctionTypes lft) g1 g2 tp_ref = if lft
    then m_app_not (m_and_2 g1 (m_app_not g2)) tp_ref
    else do
	tp <- liftIO (readIORef tp_ref)
	case tp of
	    UType Proposition _ (FunctionT n s) -> do
				mm1 <- g1 n
				mm2 <- g2 s
				return (do
					m1 <- mm1
					m2 <- mm2
					return (m1 ++ m2))
	    otherwise -> return Nothing
m_cofunction :: (MonadIO m) => UseFunctionTypes -> MatchGet m -> MatchGet m -> MatchGet m
m_cofunction (UseLowerFunctionTypes lft) g1 g2 tp_ref = if lft
    then (m_and_2 (m_app_not g1) g2) tp_ref
    else do
	tp <- liftIO (readIORef tp_ref)
	case tp of
	    UType Proposition _ (CofunctionT n s) -> do
				mm1 <- g1 n
				mm2 <- g2 s
				return (do
					m1 <- mm1
					m2 <- mm2
					return (m1 ++ m2))
	    otherwise -> return Nothing
m_and_pair :: (MonadIO m) => MatchGet m -> MatchGet m -> UTRef -> m (Maybe ([UTRef], [UTRef]))
m_and_pair g1 g2 tp_ref = do
	tp <- liftIO (readIORef tp_ref)
	case tp of
	    UType Proposition _ (AndT _ [n, s]) -> do
				mm1 <- g1 n
				mm2 <- g2 s
				return (do
					m1 <- mm1
					m2 <- mm2
					return (m1, m2))
	    otherwise -> return Nothing
m_and_2 :: (MonadIO m) => MatchGet m -> MatchGet m -> MatchGet m
m_and_2 g1 g2 tp_ref = do
	mpair <- m_and_pair g1 g2 tp_ref
	return $ fmap (\(m1,m2) -> m1 ++ m2) mpair
-- Matches like m_and_2, but returns with the operands interchanged.
m_and_swap :: (MonadIO m) => MatchGet m -> MatchGet m -> MatchGet m
m_and_swap g1 g2 tp_ref = do
	mpair <- m_and_pair g1 g2 tp_ref
	return $ fmap (\(m1,m2) -> m2 ++ m1) mpair
m_app_not :: (MonadIO m) => MatchGet m -> MatchGet m
m_app_not g1 = m_application m_not g1
m_not :: (MonadIO m) => MatchGet m
m_not tp_ref = do
	tp <- liftIO (readIORef tp_ref)
	case tp of
	    UType kind _ NotT -> m_prop_type_ctor kind
	    otherwise -> return Nothing
m_prop_name :: (MonadIO m) => VarName -> MatchGet m
m_prop_name nm = m_kinded_name nm m_prop_kind
m_ctor_name :: (MonadIO m) => VarName -> MatchGet m
m_ctor_name nm = m_kinded_name nm m_prop_type_ctor
m_kinded_name :: (MonadIO m) => VarName -> KindMatchGet m -> MatchGet m
m_kinded_name nm m_k tp_ref = do
	tp <- liftIO (readIORef tp_ref)
	case tp of
	    UType kind _ (NameT nm' _) | nm' == nm -> m_k kind
	    otherwise -> return Nothing
m_iov :: (MonadIO m) => UseIOType -> MatchGet m -> MatchGet m
m_iov (UseLowerIOType lio) g1 = if lio then m_application (m_ctor_name "IO") g1
				else m_application (m_ctor_name "IOV") g1
m_ion :: (MonadIO m) => UseIOType -> MonadModel -> MatchGet m -> MatchGet m
m_ion (UseLowerIOType lio) (UseStrictMonad sm) g1
	= if not lio then m_application (m_ctor_name "ION") g1
		else if sm then undefined -- is a problem because and ION doesn't actually match anything.
		else m_app_not (m_application (m_ctor_name "IO") (m_app_not g1))
m_monad_v :: (MonadIO m) => UseIOType -> MatchGet m -> MatchGet m -> MatchGet m
m_monad_v (UseLowerIOType lio) m g1 = if lio then m_application m g1
				else m_application m g1
m_monad_n :: (MonadIO m) => UseIOType -> MonadModel -> MatchGet m -> MatchGet m -> MatchGet m
m_monad_n (UseLowerIOType lio) (UseStrictMonad sm) m g1
	= if not lio then m_application m g1
		else if not sm then m_app_not (m_application m (m_app_not g1))
		else error "matching types with strict monads and lowering"
m_prop :: (MonadIO m) => MatchGet m
m_prop tp_ref = do
	tp <- liftIO (readIORef tp_ref)
	case tp of
	    UType Proposition _ NoT -> return $ Just []
	    otherwise -> return Nothing
m_prop_kind :: (MonadIO m) => KindMatchGet m
m_prop_kind k = return $ case k of
	Proposition -> Just []
	otherwise -> Nothing
m_prop_type_ctor :: (MonadIO m) => KindMatchGet m
m_prop_type_ctor k = case k of
	Predicate kind1_ref kind2_ref -> do
	       mk1 <- m_prop kind1_ref
	       mk2 <- m_prop kind2_ref
	       return $ do
		       k1 <- mk1
		       k2 <- mk2
		       return []
	otherwise -> return Nothing
