{-
  Ambidexter is Copyright (c) Prescott K. Turner, 2005. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
-}
module TypedAST where

import Maybe
import Control.Monad.State
import Data.Map hiding (map)
import Data.IORef
import System.IO

import Position
import Basic
import AST
import Type
import Unify
import Kind

verbose_type = True

-- AST is Typed.

class TypeStructured tp t where
    patternTypeMatches :: t -> [(VarName, (Way, tp))]
instance TypeStructured tp (Pattern' (tp, Span)) where
    patternTypeMatches (Pattern (typ,sp) pat) = case pat of
	NamePat name -> [(name, (ByValue, typ))]
	VoidPat -> []
	AndPat mpats -> concatMap submatch mpats
	    where
		submatch mpat = maybe [] patternTypeMatches mpat
	NotPat copat -> patternTypeMatches copat
	SpecificPat stp pat -> patternTypeMatches pat
	
instance KindParameterized (Pat' (UTRef, Span))
    where
	-- kindMatchesOf :: FiniteMap VarName Kind -> Pat -> [(Kind, Kind, String)]
	kindMatchesOf kmap pat = case pat of
	    VoidPat -> return []
	    AndPat mpats -> do
			matcheses <- mapM (kindMatchesOf kmap) (concatMap maybeToList mpats)
			return (concat matcheses)
	    NotPat copat -> kindMatchesOf kmap copat
	    SpecificPat tp pat -> do
			tp_matches <- kindMatchesOf kmap tp
			pat_matches <- kindMatchesOf kmap pat
			return (tp_matches ++ pat_matches)
	    otherwise -> return []
instance KindParameterized (Pattern' (UTRef, Span))
    where
	-- kindMatchesOf :: FiniteMap VarName Kind -> Pattern -> [(Kind, Kind, String)]
	kindMatchesOf kmap pattern@(Pattern typ@(tref, sp) pat) = do
		UType kind tsp tp <- liftIO $ readIORef tref
		typ_matches <- kindMatchesOf kmap (fst typ)
		pat_matches <- kindMatchesOf kmap pat
		extra_matches <- case pat of
			SpecificPat (TypeExpression (tref, _)) pat -> case tp of
					ForallT qk _ _ -> do
					    pattern_str <- do_tshow TopShowSpan True True pattern
					    return [(tref, qk, pattern_str)]
					otherwise -> return []
			otherwise -> return []
		return $ typ_matches ++ pat_matches ++ extra_matches

instance Typed (Pattern' (UTRef, Span))
    where
	typeMatchesOf mm uio ufn tmap pattern@(Pattern (tp,sp) pat) = case pat of
		NamePat nm -> return []
		VoidPat -> do
			name_tref <- liftIO $ newIORef (UType Proposition sp $ NameT "True" "")
			fin <- finish (tp, name_tref, (\tp -> \_ -> "The type " ++ tp ++ " must match the void pattern's True."))
			return [fin]
		AndPat mpats -> do
			pat_matches <- mapM productElemMatch mpats
			and_tref <- liftIO $ newIORef (UType Proposition sp $ AndT PlainOrigin pat_matches)
			other_matches <- mapM (typeMatchesOf mm uio ufn tmap) (concatMap maybeToList mpats)
			fin <- finish (tp, and_tref, (\tp -> \andt -> "The type " ++ tp ++ " must match the and pattern's " ++ andt ++ "."))
			return $ [fin] ++ concat other_matches
		NotPat copattern@(Copattern (ctp,sp) copat) -> do
			notc <- uctorT_ sp NotT
			not_tref <- liftIO $ newIORef (UType Proposition sp $ ApplicationT notc ctp)
			other_matches <- typeMatchesOf mm uio ufn tmap copattern
			fin <- finish (tp, not_tref, (\tp -> \nott -> "The type " ++ tp ++ " must match the not pattern's " ++ nott ++ "."))
			return $ [fin] ++ other_matches
		SpecificPat (TypeExpression (sub_stp,_)) sub_pattern@(Pattern (sub_ptp,sub_sp) sub_pat) -> do
			other_matches <- typeMatchesOf mm uio ufn tmap sub_pattern
			let satisfy = do
				tpval <- liftIO $ readIORef tp
				case tpval of
				    UType k tp_sp (ForallT k' tnm tp') -> do
					tp2 <- substituteTypeName tnm sub_stp tp'
					fin <- finish (sub_ptp, tp2, \subt -> \tp2 -> "In the specific pattern, the subpattern type " ++ subt ++ " must match the substituted body of the forall, " ++ tp2 ++ "." )
					return (Just fin)
				    otherwise -> return Nothing
			(m_fin :: Maybe (TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)) <- satisfy
			fin <- case m_fin of
			    Just fin -> return fin
			    Nothing -> do
				pat_str <- do_tshow TopShowSpan False False pattern
				let unsatisfied_tp = UnsatisfiedT satisfy $ "Universal type of pattern " ++ pat_str ++ " could not be deduced."
				unsatisfied_tref <- liftIO $ newIORef $ UType Proposition sub_sp $ unsatisfied_tp
				fin <- finish (sub_ptp, unsatisfied_tref, (\subt -> \_ -> "In the specific pattern, the subpattern type " ++ subt ++ " must be universal."))
				return fin
			return $ [fin] ++ other_matches
	    where
		finish (tp1,tp2, msgf) = do
		    let pat_shower show_kinds tp1_str tp2_str = do
			pat_str <- do_tshow TopShowSpan verbose_type show_kinds pattern
			let msg = msgf tp1_str tp2_str
			return $ "Type mismatch at pattern: " ++ pat_str ++ "\n" ++ msg
		    return (TypeInContext tp1 tmap, TypeInContext tp2 tmap, pat_shower)
		productElemMatch mpat = case mpat of Just (Pattern (ptp,sp) _) -> return ptp
						     Nothing -> prep (Type Proposition sp NoT)

instance Typed (Term' (UTRef, Span))
    where
	typeMatchesOf mm uio ufn tmap term@(TM (tp,sp) op) =  case (op :: Op' (UTRef, Span)) of
		ListTerm ls -> do
			list_tp <- prep $ listT sp (Type Proposition sp NoT)
			list_matches <- mapM listElemMatch ls
			let msgf tp lst = "The type " ++ tp ++ " must match the list's " ++ lst ++ "."
			finish $ (tp, list_tp, msgf) : list_matches
		VariableTerm str -> maybe (die $ "Variable '" ++ str ++ "' is not in scope at " ++ show sp ++ ".")
					(\(way, tic) -> do
						let op_str = show sp
						let op_shower show_kinds tp tic = do
						    return $ "Type mismatch at " ++ op_str ++ ".\nThe context type " ++ tp ++ " must match the type inferred elsewhere for the name '" ++ str ++ "', " ++ tic ++ "."
						assertWay ByValue way op_str (return [(TypeInContext tp tmap, tic, op_shower)]))
					(Data.Map.lookup str tmap)
		IntegerTerm n -> do
			name_tp <- prep $ Type Proposition sp $ NameT "Integer" ""
			finish [(tp, name_tp, \tp -> \_ -> "The type " ++ tp ++ " must match the inter literal's Integer.")]
		StringTerm str -> do
			name_tp <- prep $ Type Proposition sp $ NameT "String" ""
			finish [(tp, name_tp, \tp -> \_ -> "The type " ++ tp ++ " must match the string literal's String.")]
		LambdaTerm (Pattern (ptp,psp) pat) (TM (ttp,tsp) _) -> do
			let UseLowerFunctionTypes lfn = ufn
			function_tp <- if not lfn then liftIO $ newIORef $ UType Proposition sp $ FunctionT ptp ttp
			    else do
				tnotc <- uctorT_ tsp NotT
				nttp <- liftIO $ newIORef $ UType Proposition tsp $ ApplicationT tnotc ttp
				atp <- liftIO $ newIORef $ UType Proposition sp $ AndT FunctionOrigin [ptp, nttp]
				notc <- uctorT_ sp NotT
				liftIO $ newIORef $ UType Proposition sp $ ApplicationT notc atp
			finish [(tp, function_tp, \tp -> \ftp -> "The type " ++ tp ++ " must match the lambda term's function type " ++ ftp ++ ".")]
		-- BindTerm (TM (Type Proposition _ (ApplicationT (Type _ _ (NameT "IOV" "")) s),_) act) (TM (Type Proposition _ (FunctionT s' (Type Proposition _ (ApplicationT (Type _ _ (NameT "IOV" "")) t))),bsp) binder) -> [(s,s'), (tp, iOVT sp t)] -- two stage unification
		BindTerm (TM (atp_ref,asp) act) (TM (btp_ref,bsp) binder) -> do
			let satisfy = do
			    matps <- m_monad_v uio m_any m_any atp_ref
			    mbtps <- m_function ufn m_any (m_monad_v uio m_any m_any) btp_ref
			    maybe_do $ do
				[m, s] <- matps
				[s', m', t] <- mbtps
				return $ do
				    iovt_ref <- u_mVT uio sp m t
				    finish [(m,m', \ma -> \mb -> "The left type constructor " ++ ma ++ " must match the right type constructor " ++ mb ++ "."),
					    (s,s', \sa -> \sb -> "The type returned by the left, " ++ sa ++ " must match the type accepted by the right, " ++ sb ++ "."),
					    (tp, iovt_ref, \tp -> \io -> "The type " ++ tp ++ " must match the binding term's " ++ io ++ ".")] -- two stage unification
			m_fin <- satisfy
			case m_fin of
			    Just fins -> return fins
			    Nothing -> do
				let UseLowerIOType lio = uio
				let ioized = applicationT_ sp
				atp' <- prep ioized
				btp' <- prep $ functionT ufn sp (Type Proposition sp NoT) ioized
				let sat selector = do
					m_satisfiers <- satisfy
					return $ case m_satisfiers of
						Just fin -> Just (selector fin)
						otherwise -> Nothing
				let unsatisfied_tp sel = UnsatisfiedT (sat sel) $ "Internal error: Bind unification failed."
				let unsatisfied_tref sel = do
					u <- liftIO $ newIORef $ UType Proposition sp $ (unsatisfied_tp sel)
					return (tp, u, \_-> \_->"Internal error: Bind unification failed.")
				ut1 <- unsatisfied_tref (!! 0)
				ut2 <- unsatisfied_tref (!! 1)
				ut3 <- unsatisfied_tref (!! 2)
				finish [(atp_ref, atp', \tp-> \_->"The type of the binding's left, " ++ tp ++ " must apply a type constructor."),
					(btp_ref, btp', \tp-> \_->"The type of the binding's right, " ++ tp ++ " must be a function returning the application of a type constructor."),
					ut1, ut2, ut3]
				
		-- SeqTerm (TM (Type Proposition _ (ApplicationT (Type _ _ (NameT "IOV" "")) s),asp) act) (TM (Type Proposition _ (ApplicationT (Type _ _ (NameT "IOV" "")) t),bsp) binder) -> [(tp, iOVT sp t)] -- two stage unifiction
		SeqTerm (TM (atp_ref,asp) act) (TM (btp_ref,bsp) binder) -> do
			let satisfy = do
			    matps <- m_monad_v uio m_any m_any atp_ref
			    mbtps <- m_monad_v uio m_any m_any btp_ref
			    maybe_do $ do
				[m, s] <- matps
				[m', t] <- mbtps
				return $ do
				    iovt_ref <- u_mVT uio sp m t
				    finish [(m,m', \ma -> \mb -> "The left type constructor " ++ ma ++ " must match the right type constructor " ++ mb ++ "."),
					    (tp, iovt_ref, \tp -> \io -> "The type " ++ tp ++ " must match the sequencing term's " ++ io ++ ".")] -- two stage unification
			m_fin <- satisfy
			case m_fin of
			    Just fins -> return fins
			    Nothing -> do
				let UseLowerIOType lio = uio
				let ioized = applicationT_ sp
				atp' <- prep ioized
				btp' <- prep ioized
				let sat selector = do
					m_satisfiers <- satisfy
					return $ case m_satisfiers of
						Just fin -> Just (selector fin)
						otherwise -> Nothing
				let unsatisfied_tp sel = UnsatisfiedT (sat sel) $ "Internal error: Seq unification failed."
				let unsatisfied_tref sel = do
					u <- liftIO $ newIORef $ UType Proposition sp $ (unsatisfied_tp sel)
					return (tp, u, \_-> \_->"Internal error : Seq unification failed.")
				ut1 <- unsatisfied_tref (!! 0)
				ut2 <- unsatisfied_tref (!! 1)
				finish [(atp_ref, atp', \tp-> \_->"The type on the sequence's left, " ++ tp ++ " must apply a type constructor."),
					(btp_ref, btp', \tp-> \_->"The type on the sequence's right, " ++ tp ++ " must apply a type constructor."),
					ut1, ut2]
		ApplicationTerm (TM (atp,asp) arg) (TM (ftp,fsp) func) -> do
			let UseLowerFunctionTypes lfn = ufn
			let tsp = sp
			function_tref <- if not lfn then liftIO $ newIORef $ UType Proposition sp $ FunctionT atp tp
			    else do
				tnotc <- uctorT_ tsp NotT
				ntp <- liftIO $ newIORef $ UType Proposition tsp $ ApplicationT tnotc tp
				atp <- liftIO $ newIORef $ UType Proposition sp $ AndT FunctionOrigin [atp, ntp]
				notc <- uctorT_ sp NotT
				liftIO $ newIORef $ UType Proposition sp $ ApplicationT notc atp
			finish [(ftp, function_tref, \ftp -> \itp -> "The type on the right " ++ ftp ++ " must be the function type matching its context, " ++ itp ++ ".")]
		VoidTerm -> do
			name_tref <- liftIO (newIORef (UType Proposition sp $ NameT "True" ""))
			finish [(tp, name_tref, \tp-> \_->"The type " ++ tp ++ " of the void term must be True.")]
		ProductTerm ts -> do
			and_tref <- liftIO (newIORef (UType Proposition sp $ AndT PlainOrigin $ map productElemMatch ts))
			finish [(tp, and_tref, \tp-> \tand->"The type " ++ tp ++ " of the tuple must match the constructed 'and' type " ++ tand ++ ".")]
		PlanTerm (TO (ctp,csp) c) -> do
			notc <- uctorT_ sp NotT
			not_tref <- liftIO (newIORef (UType Proposition sp $ ApplicationT notc ctp))
			finish [(tp, not_tref, \tp-> \tnot-> "The type " ++ tp ++ " from context must match the 'not' type " ++ tnot ++ " inferred from within.")]
		AbstractTerm (Copattern (ptp,psp) copat) (TM (ttp,tsp) t) (TO (ctp,csp) c) ->
			finish [(ttp,ctp, \ttp-> \ctp->"The abstraction's left type " ++ ttp ++ " must match its right type " ++ ctp ++ "."),
				(tp, ptp, \tp-> \ptp->"The type " ++ tp ++ " must match its pattern's constructed type " ++ ptp ++ ".")]
		AnnotatedTerm preu (TM (ttp,tsp) t) -> finish' (not preu) [(tp, ttp, \tp -> \ttp -> "The annotation " ++ tp ++ " must match the type of the contained term " ++ ttp ++ ".")]
		GenericTypeTerm (tref,_) nm (TM (ttp,tsp) t) -> do
			-- UType kind _ _ <- liftIO (readIORef tref)
			let kind = tref   -- Share the kind of the GenericTypeTerm.
			forall_tref <- liftIO (newIORef (UType Proposition sp $ ForallT tref nm ttp))
			finish [(tp, forall_tref, \tp-> \all-> "The type " ++ tp ++ " must match the universal type constructed from its contained term " ++ all ++ ".")]
	    where
		finish :: [(UTRef, UTRef, String -> String -> String)] -> StateT [TypeId] IO [(TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)]
		finish = finish' True
		finish' go_deeper local_matches = do
		    let op_shower msgf show_kinds tp1_str tp2_str = do
			op_str <- do_tshow TopShowSpan verbose_type show_kinds term
			let msg = msgf tp1_str tp2_str
			return $ "Type mismatch at term: " ++ op_str ++ "\n" ++ msg
		    deepers <- if go_deeper then deeper_matches else return []
		    return $ (map (\(tp1,tp2,msgf) -> (TypeInContext tp1 tmap, TypeInContext tp2 tmap, op_shower msgf)) local_matches) ++ deepers
		listElemMatch (TM (ttp,tsp) _) = do
		    list_tp <- ulistT sp ttp
		    return (tp, list_tp, \tp -> \list -> "The type of the list element, " ++ tp ++ ", must match the type of the list as a whole, " ++ list ++ ".")
		productElemMatch (TM (ttp,tsp) _) = ttp
		tmap' = case op of
			LambdaTerm pat term -> insertPatternTypes tmap pat
			AbstractTerm copat t c -> insertPatternTypes tmap copat
			GenericTypeTerm (kref,_) nm _ -> addToTypeContext tmap nm (ByType, kref)
			otherwise -> tmap
		deeper_matches = do
			subex_match_list <- mapM (typeMatchesOf mm uio ufn tmap') (listSubexpressions op)
			subco_match_list <- mapM (typeMatchesOf mm uio ufn tmap') (listSubcoexpressions op)
			other_matches <- case op of
						LambdaTerm pat t -> typeMatchesOf mm uio ufn tmap pat
						AbstractTerm pat t c -> typeMatchesOf mm uio ufn tmap pat
						otherwise -> return []
			return (concat subex_match_list ++ concat subco_match_list ++ other_matches)

instance KindParameterized (Op' (UTRef, Span)) where
	kindMatchesOf kmap op = do
		matcheses <- mapM (kindMatchesOf kmap) (listSubexpressions op)
		comatcheses <- mapM (kindMatchesOf kmap) (listSubcoexpressions op)
		extra_matches <- case op of
				LambdaTerm pat t -> kindMatchesOf kmap pat
				AbstractTerm pat t c -> kindMatchesOf kmap pat
				otherwise -> return []
		return $ concat matcheses ++ concat comatcheses ++ extra_matches

instance KindParameterized (Term' (UTRef, Span))
    where
	kindMatchesOf kmap term@(TM typ@(tref, sp) op) = do
		UType kind _ tp <- liftIO $ readIORef tref
		local_matches <- case op of
		    GenericTypeTerm (tref,_) nm t -> do
			-- UType gk _ _ <- liftIO $ readIORef tref
			case tp of
			    ForallT qk _ _ -> do
				term_str <- do_tshow TopShowSpan True True term
				return [(tref, qk, term_str)]
			    otherwise -> return []
		    otherwise -> return []
		typ_matches <- kindMatchesOf kmap (fst typ)
		kmap' <- case op of
		    GenericTypeTerm (tref,_) nm t -> do
			-- UType kind _ _ <- liftIO $ readIORef tref
			return $ Data.Map.insert nm tref kmap
		    op' -> return kmap
		op_matches <- kindMatchesOf kmap' op
	        return $ local_matches ++ typ_matches ++ op_matches

instance TypeStructured tp (Copattern' (tp, Span)) where
    patternTypeMatches (Copattern (typ,sp) copat) = case copat of
	NameCopat name -> [(name, (ByName, typ))]
	UnreachedCopat -> []
	OrCopat mcopats -> concatMap submatch mcopats
	    where
		submatch mcopat = maybe [] patternTypeMatches mcopat
	NotCopat pat -> patternTypeMatches pat
	PackageCopat stp copat -> patternTypeMatches copat

instance KindParameterized (Copat' (UTRef, Span))
    where
	-- kindMatchesOf :: FiniteMap VarName Kind -> Copat -> [(Kind, Kind, String)]
	kindMatchesOf kmap copat = case copat of
	    OrCopat mcopats -> do
			matcheses <- mapM (kindMatchesOf kmap) (concatMap maybeToList mcopats)
			return (concat matcheses)
	    NotCopat pat -> kindMatchesOf kmap pat
	    PackageCopat tp copat -> do
			tp_matches <- kindMatchesOf kmap tp
			copat_matches <- kindMatchesOf kmap copat
			return (tp_matches ++ copat_matches)
	    otherwise -> return []
instance KindParameterized (Copattern' (UTRef, Span))
    where
	-- kindMatchesOf :: FiniteMap VarName Kind -> Copattern -> [(Kind, Kind, String)]
	kindMatchesOf kmap copattern@(Copattern typ@(tref, sp) copat) = do
		UType kind _ tp <- liftIO $ readIORef tref
		typ_matches <- kindMatchesOf kmap (fst typ)
		copat_matches <- kindMatchesOf kmap copat
		extra_matches <- case copat of
			PackageCopat (TypeExpression (tref, _)) copat -> do
				-- UType sk _ stp <- liftIO $ readIORef tref
				case tp of
					ExistsT qk _ _ -> do
					    copattern_str <- do_tshow TopShowSpan True True copattern
					    return [(tref, qk, copattern_str)]
					otherwise -> return []
			otherwise -> return []
		return $ typ_matches ++ copat_matches ++ extra_matches

instance Typed (Copattern' (UTRef, Span))
    where
	typeMatchesOf mm uio ufn tmap copattern@(Copattern (tp,sp) copat) = case copat of
		NameCopat nm -> return []
		UnreachedCopat -> do
			name_tref <- liftIO $ newIORef (UType Proposition sp $ NameT "False" "")
			fin <- finish (tp, name_tref, (\tp-> \_->"The type " ++ tp ++ " must match the unreached copattern's False."))
			return [fin]
		OrCopat mcopats -> do
			pat_matches <- mapM productElemMatch mcopats 
			or_tref <- liftIO $ newIORef (UType Proposition sp $ OrT pat_matches)
			other_matches <- mapM (typeMatchesOf mm uio ufn tmap) (concatMap maybeToList mcopats)
			fin <- finish (tp, or_tref, \tp-> \ort->"The type " ++ tp ++ " must match the or pattern's " ++ ort ++ ".")
			return $ [fin] ++ concat other_matches
		NotCopat pattern@(Pattern (ptp,psp) pat) -> do
			notc <- uctorT_ sp NotT
			not_tref <- liftIO $ newIORef (UType Proposition sp $ ApplicationT notc ptp)
			other_matches <- typeMatchesOf mm uio ufn tmap pattern
			fin <- finish (tp, not_tref, \tp-> \tnot-> "The type " ++ tp ++ " must match the not copattern's " ++ tnot ++ ".")
			return $ [fin] ++ other_matches
		PackageCopat (TypeExpression (sub_stp,_)) sub_copattern@(Copattern (sub_ptp,sub_sp) sub_copat) -> do
			other_matches <- typeMatchesOf mm uio ufn tmap sub_copattern
			let satisfy = do
				tpval <- liftIO $ readIORef tp
				case tpval of
				    UType k tp_sp (ExistsT k' tnm tp') -> do
					tp2 <- substituteTypeName tnm sub_stp tp'
					fin <- finish (sub_ptp, tp2, \subt -> \tp2 -> "In the package copattern, the subpattern type " ++ subt ++ " must match the substituted body of the forall, " ++ tp2 ++ ".")
					return (Just fin)
				    otherwise -> return Nothing
			(m_fin :: Maybe (TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)) <- satisfy
			fin <- case m_fin of
			    Just fin -> return fin
			    Nothing -> do
				pat_str <- do_tshow TopShowSpan False False copattern
				let unsatisfied_tp = UnsatisfiedT satisfy $ "Existential type of pattern " ++ pat_str ++ " could not be deduced."
				unsatisfied_tref <- liftIO $ newIORef $ UType Proposition sub_sp $ unsatisfied_tp
				fin <- finish (sub_ptp, unsatisfied_tref, \subt -> \_ -> "In the package pattern, the subpattern type " ++ subt ++ " must be existential.")
				return fin
			return $ [fin] ++ other_matches
	    where
		finish (tp1,tp2, msgf) = do
		    let pat_shower show_kinds tp1_str tp2_str = do
			pat_str <- do_tshow TopShowSpan verbose_type show_kinds copattern
			let msg = msgf tp1_str tp2_str
			return $ "Type mismatch at copattern: " ++ pat_str ++ "\n" ++ msg
		    return (TypeInContext tp1 tmap, TypeInContext tp2 tmap, pat_shower)
		productElemMatch mcopat = case mcopat of
							Just (Copattern (ptp,sp) _) -> return ptp
							Nothing -> prep (Type Proposition sp NoT)

instance Typed (Coterm' (UTRef, Span))
    where
	typeMatchesOf mm uio ufn tmap term@(TO (typ, sp) op) = case op of
		ColistTerm ls -> do
			list_tp <- prep $ colistT sp (Type Proposition sp NoT)
			list_matches <- mapM listElemMatch ls
			let msgf tp lst = "The type " ++ tp ++ " must match the colist's " ++ lst ++ "."
			finish $ (typ, list_tp, msgf) : list_matches
		CovariableTerm str -> maybe (die $ "Covariable '" ++ str ++ "' is not in scope at " ++ show sp ++ ".")
					(\(way, tic) -> do
						let op_str = show sp
						let op_shower show_kinds tic tp = do
						    return $ "Type mismatch at " ++ op_str ++ ".\nThe context type " ++ tp ++ " must match the type inferred elsewhere for the name '" ++ str ++ "', " ++ tic ++ "."
						assertWay ByName way op_str (return [(tic, TypeInContext typ tmap, op_shower)]))
					(Data.Map.lookup str tmap)
		ColambdaTerm (Copattern (ptp,psp) copat) (TO (ttp,tsp) _) -> do
			let UseLowerFunctionTypes lfn = ufn
			cofunction_tp <- if not lfn then liftIO $ newIORef $ UType Proposition sp $ CofunctionT ptp ttp
			    else do
				notc <- uctorT_ psp NotT
				nptp <- liftIO $ newIORef $ UType Proposition psp $ ApplicationT notc ptp
				liftIO $ newIORef $ UType Proposition sp
				    $ AndT CofunctionOrigin [nptp, ttp]
			finish [(typ, cofunction_tp, \tp -> \ftp -> "The type " ++ tp ++ " must match the colambda term's cofunction type " ++ ftp ++ ".")]
		-- CobindTerm (TO (Type Proposition _ (ApplicationT (Type _ _ (NameT "ION" "")) s),asp) act) (TO (Type Proposition _ (CofunctionT s' (Type Proposition _ (ApplicationT (Type _ _ (NameT "ION" "")) t))),bsp) binder) -> [(s,s'), (typ, iONT sp t)] -- two stage unification
		CobindTerm (TO (atp_ref,asp) act) (TO (btp_ref,bsp) binder) -> do
			let satisfy = do
			    -- [x->([z->x$[(Bind z func)]])$arg]   x:~!t  z:!s  func:s&~!t  arg:~!s
			    --              Bind z func:!t
			    let UseLowerIOType lio = uio
			    let UseStrictMonad sm = mm
			    let UseLowerFunctionTypes lft = ufn
			    matps <- (if lio then m_app_not (m_application m_any m_any)
					else m_monad_n uio mm m_any m_any) atp_ref
			    mbtps <- (if lio && lft then m_and_2
							(if not sm then m_any
							    else m_app_not (m_app_not m_any))
							(m_app_not (m_application m_any m_any))
					else m_cofunction ufn m_any (m_monad_n uio mm m_any m_any)) btp_ref
			    maybe_do $ do
				[m, s] <- matps
				[s', m', t] <- mbtps
				return $ do
				    iont_ref <- if lio then do
							iotref <- u_mT sp m t
							notc_tref <- uctorT_ sp NotT
							u_mT sp notc_tref iotref
						else u_mNT uio sp m t
				    finish [(m,m', \ma -> \mb -> "The left type constructor " ++ ma ++ " must match the right type constructor " ++ mb ++ "."),
					    (s,s', \sa -> \sb -> "The type returned by the left, " ++ sa ++ "must match the type accepted by the right, " ++ sb ++ "."),
					    (typ, iont_ref, \tp -> \io -> "The type " ++ tp ++ " must match the binding coterm's " ++ io ++ ".")] -- two stage unifiction
			m_fin <- satisfy
			case m_fin of
			    Just fins -> return fins
			    Nothing -> do
				let UseLowerIOType lio = uio
				let UseLowerFunctionTypes lft = ufn
				let UseStrictMonad sm = mm
				let ioized = if lio then notT sp (if sm then applicationT_ sp
							  else applicationT sp (notT sp (Type Proposition sp NoT)))
						else applicationT_ sp
				atp' <- prep ioized
				btp' <- prep $ if lio && lft then andT sp CofunctionOrigin
							[if not sm then noT else notT sp (notT sp noT), ioized]
						else cofunctionT ufn sp (Type Proposition sp NoT) ioized
				let sat selector = do
					m_satisfiers <- satisfy
					return $ case m_satisfiers of
						Just fin -> Just (selector fin)
						otherwise -> Nothing
				let unsatisfied_tp sel = UnsatisfiedT (sat sel) $ "Internal error: Cobind unification failed."
				let unsatisfied_tref sel = do
					u <- liftIO $ newIORef $ UType Proposition sp $ (unsatisfied_tp sel)
					return (typ, u, \_-> \_->"Internal error: Cobind unification failed.")
				ut1 <- unsatisfied_tref (!! 0)
				ut2 <- unsatisfied_tref (!! 1)
				ut3 <- unsatisfied_tref (!! 2)
				finish [(atp_ref, atp', \tp-> \_->"The type of the cobinding's left, " ++ tp ++ " must apply a type constructor."),
					(btp_ref, btp', \tp-> \_->"The type of the cobinding's right, " ++ tp ++ " must be a function returning the application of a type constructor."), ut1, ut2, ut3]
				
		-- CoseqTerm (TO (Type Proposition _ (ApplicationT (Type _ _ (NameT "ION" "")) s),asp) act) (TO (Type Proposition _ (ApplicationT (Type _ _ (NameT "ION" "")) t),bsp) binder) -> [(typ, iONT sp t)] -- two stage unifiction
		CoseqTerm (TO (atp_ref,asp) act) (TO (btp_ref,bsp) binder) -> do
			let satisfy = do
			    let UseLowerIOType lio = uio
			    matps <- (if lio then m_app_not (m_application m_any m_any)
					else m_monad_n uio mm m_any m_any) atp_ref
			    mbtps <- (if lio then m_app_not (m_application m_any m_any)
					else m_monad_n uio mm m_any m_any) btp_ref
			    maybe_do $ do
				[m, s] <- matps
				[m', t] <- mbtps
				return $ do
				    iont_ref <- if lio then return btp_ref
						else u_mNT uio sp m t
				    finish [(m, m', \ma -> \mb -> "The left type constructor " ++ ma ++ " must match the right type constructor " ++ mb ++ "."),
					    (typ, iont_ref, \tp-> \io->"The type " ++ tp ++ " must match the cosequencing term's " ++ io ++ ".")] -- two stage unifiction
			m_fin <- satisfy
			case m_fin of
			    Just fins -> return fins
			    Nothing -> do
				let UseLowerIOType lio = uio
				    UseLowerFunctionTypes lfn = ufn
				let UseStrictMonad sm = mm
				let ioized = if lio then notT sp (if sm then applicationT_ sp
							  else applicationT sp (notT sp (Type Proposition sp NoT)))
						else applicationT_ sp
				atp' <- prep ioized 
				btp' <- prep ioized
				let sat selector = do
					m_satisfiers <- satisfy
					return $ case m_satisfiers of
						Just fin -> Just (selector fin)
						otherwise -> Nothing
				let unsatisfied_tp sel = UnsatisfiedT (sat sel) $ "Internal error: Coseq unification failed."
				let unsatisfied_tref sel = do
					u <- liftIO $ newIORef $ UType Proposition sp $ (unsatisfied_tp sel)
					return (typ, u, \_-> \_->"Internal error: Coseq unification failed.")
				ut1 <- unsatisfied_tref (!! 0)
				ut2 <- unsatisfied_tref (!! 1)
				finish [(atp_ref, atp', \tp-> \_->"The type on the cosequence's left, " ++ tp ++ " must apply a type constructor."),
					(btp_ref, btp', \tp-> \_->"The type on the cosequence's right, " ++ tp ++ " must apply a type constructor."),
					ut1, ut2]
		CoapplicationTerm (TO (atp,asp) arg) (TO (ftp,fsp) func) -> do
			let UseLowerFunctionTypes lfn = ufn
			cofunction_tref <- if not lfn then liftIO (newIORef $ UType Proposition sp $ CofunctionT atp typ)
			    else do
				notc <- uctorT_ asp NotT
				nptp <- liftIO $ newIORef $ UType Proposition asp $ ApplicationT notc atp
				liftIO $ newIORef $ UType Proposition sp
				    $ AndT CofunctionOrigin [nptp, typ]
			finish [(ftp, cofunction_tref, \ftp -> \itp -> "The type on the right " ++ ftp ++ " must be the function type matching its context, " ++ itp ++ ".")]
		UseTerm (TM (ntp,nsp) ntm) -> do
			notc <- uctorT_ sp NotT
			not_tref <- liftIO (newIORef (UType Proposition sp (ApplicationT notc ntp)))
			finish [(typ, not_tref, \typ-> \tnot -> "The type " ++ typ ++ " must match the 'not' type " ++ tnot ++ " inferred from within.")]
		UnreachedTerm -> do
			name_tref <- liftIO (newIORef (UType Proposition sp $ NameT "False" ""))
			finish [(typ, name_tref, \typ-> \_->"The type " ++ typ ++ " of the unreached coterm must be False.")]
		SumTerm cs -> do
			or_tref <- liftIO (newIORef (UType Proposition sp $ OrT $ map sumElemMatch cs))
			finish [(typ, or_tref, \tp-> \tor->"The type " ++ tp ++ " of the case must match the constructed 'or' type " ++ tor ++ ".")]
		CoabstractTerm (Pattern (ptp,psp) pat) (TM (ttp,tsp) t) (TO (ctp,csp) c) ->
			finish [(ttp,ctp, \ttp-> \ctp->"The coabstraction's left type " ++ ttp ++ " must match its right type " ++ ctp ++ "."),
				(typ, ptp, \tp-> \ptp->"The type " ++ tp ++ " must match its pattern's construted type " ++ ptp ++ ".")]
		CoannotatedTerm preu (TO (ttp,tsp) c) ->
			finish' (not preu) [(typ, ttp, \tp-> \ttp -> "The annotation " ++ tp ++ " must match the type of the contained coterm " ++ ttp ++ ".")]
		AbstractTypeTerm (tref,_) nm (TO (ttp,tsp) t) -> do
			-- UType kind _ _ <- liftIO (readIORef tref)
			let kind = tref  -- Share the kind of the AbstractTypeTerm.
			exists_tref <- liftIO (newIORef (UType Proposition sp $ ExistsT kind nm ttp))
			finish [(typ, exists_tref, \tp-> \ex-> "The type " ++ tp ++ " must match the existential type construted from its contained coterm " ++ ex ++ ".")]
	    where
		finish :: [(UTRef, UTRef, String -> String -> String)] -> StateT [TypeId] IO [(TypeInContext, TypeInContext, Bool -> String -> String -> StateT [TypeId] IO String)]
		finish = finish' True
		finish' go_deeper local_matches = do
		    let op_shower msgf show_kinds tp1_str tp2_str = do
			op_str <- do_tshow TopShowSpan verbose_type show_kinds term
			let msg = msgf tp1_str tp2_str
			return $ "Type mismatch at coterm: " ++ op_str ++ "\n" ++ msg
		    deepers <- if go_deeper then deeper_matches else return []
		    return $ map (\(tp1,tp2,msgf) -> (TypeInContext tp1 tmap, TypeInContext tp2 tmap, op_shower msgf)) local_matches ++ deepers
		listElemMatch (TO (ttp,tsp) _) = do
		    list_tp <- ucolistT sp ttp
		    return (typ, list_tp, \tp -> \list -> "The type of the colist element, " ++ tp ++ ", must match the type of the colist as a whole, " ++ list ++ ".")
		sumElemMatch (TO (ttp,tsp) _) = ttp
		tmap' = case op of
			CoabstractTerm pat t c -> insertPatternTypes tmap pat
			ColambdaTerm copat term -> insertPatternTypes tmap copat
			AbstractTypeTerm (kref,_) nm _ -> addToTypeContext tmap nm (ByType, kref)
			otherwise -> tmap
		deeper_matches = do
			subco_match_list <- mapM (typeMatchesOf mm uio ufn tmap') (listSubcoexpressions op)
			subex_match_list <- mapM (typeMatchesOf mm uio ufn tmap') (listSubexpressions op)
			other_matches <- case op of
						ColambdaTerm pat c -> typeMatchesOf mm uio ufn tmap pat
						CoabstractTerm pat t c -> typeMatchesOf mm uio ufn tmap pat
						otherwise -> return []
			return (concat subco_match_list ++ concat subex_match_list ++ other_matches)

insertPatternTypes :: (TypeStructured UTRef (t (UTRef, Span))) => TypeContext -> t (UTRef, Span) -> TypeContext
insertPatternTypes tmap pat
	= maybe (foldr insert1 tmap matches) (\dupe -> error ("\"" ++ dupe ++ "\" is duplicated in pattern")) possible_duplicate
    where
        insert1 (nm, waytp) tpmap = addToTypeContext tpmap nm waytp
	matches = patternTypeMatches pat
	binding_names = map fst matches
	possible_duplicate = find_duplicate binding_names
	find_duplicate ls = case ls of
		(n:ns) -> if n `elem` ns then Just n
				else find_duplicate ns
		[] -> Nothing

instance KindParameterized (Coop' (UTRef, Span)) where
	kindMatchesOf kmap coop = do
		matcheses <- mapM (kindMatchesOf kmap) (listSubexpressions coop)
		comatcheses <- mapM (kindMatchesOf kmap) (listSubcoexpressions coop)
		extra_matches <- case coop of
				ColambdaTerm pat c -> kindMatchesOf kmap pat
				CoabstractTerm pat t c -> kindMatchesOf kmap pat
				otherwise -> return []
		return $ concat matcheses ++ concat comatcheses ++ extra_matches

instance KindParameterized (Coterm' (UTRef, Span))
    where
	kindMatchesOf kmap coterm@(TO (tref, sp) coop) = do
		typ@(UType kind _ tp) <- liftIO $ readIORef tref
		local_matches <- case coop of
		    AbstractTypeTerm (tref,_) nm t -> do
			-- UType ak _ _ <- liftIO $ readIORef tref
			let ak = tref
			case tp of
			    ExistsT qk _ _ -> do
				coterm_str <- do_tshow TopShowSpan True True coterm
				return [(ak, qk, coterm_str)]
			    otherwise -> return []
		    otherwise -> return []
		typ_matches <- kindMatchesOf kmap tref
		kmap' <- case coop of
		    AbstractTypeTerm (tref,_) nm t -> do
			-- UType kind _ _ <- liftIO $ readIORef tref
			let kind = tref
			return $ Data.Map.insert nm kind kmap
		    op' -> return kmap
		coop_matches <- kindMatchesOf kmap' coop
	        return $ local_matches ++ typ_matches ++ coop_matches
instance KindParameterized (TypeExpression' (UTRef, Span))
    where
	kindMatchesOf kmap term@(TypeExpression (typ, sp))
	        = kindMatchesOf kmap typ

instance (KindParameterized tp) => KindParameterized (tp, Span) where
    kindMatchesOf kmap (tp,sp) = kindMatchesOf kmap tp
instance (Typed (exprt (UTRef, Span))) => Typed (XExpr exprt UTRef) where
	typeMatchesOf mm uio ufn fm (XExpr trm) = typeMatchesOf mm uio ufn fm trm

instance (KindParameterized(exprt (UTRef, Span))) => KindParameterized (XExpr exprt UTRef) where
	kindMatchesOf kfm (XExpr expr) = kindMatchesOf kfm expr
