/*
  harmonograph is Copyright (c) Prescott K. Turner, 2006. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
*/
#include "canvas.h"
#include "frame.h"
#include "events.h"
#include "pendulums.h"
#include "trialdoc.h"

#include <unistd.h>

using namespace std;

BEGIN_EVENT_TABLE(Frame, wxDocChildFrame)
    EVT_MENU(ID_Quit, Frame::OnQuit)
    EVT_MENU(ID_Export, Frame::OnExport)
    EVT_BUTTON(ID_Start_Button, Frame::OnStart)
    EVT_BUTTON(ID_Stop_Button, Frame::OnStop)
    EVT_BUTTON(ID_Back_Button, Frame::OnBack)
    EVT_TIMER(ID_Tick, Frame::OnTimer)
END_EVENT_TABLE()

Frame::Frame(wxDocument* doc, wxView* view, wxFrame* parent, const wxString& title, const wxPoint& pos, const wxSize& size)
  : wxDocChildFrame(doc, view, parent, -1, title, pos, size),
    m_view(view),
    timer(this, ID_Tick)
{
    // Make a menubar
    menuFile = new wxMenu;

    menuFile->Append(wxID_NEW, _T("&New..."));
    menuFile->Append(wxID_OPEN, _T("&Open..."));
    menuFile->Append(wxID_CLOSE, _T("&Close"));
    menuFile->Append(wxID_SAVE, _T("&Save"));
    menuFile->Append(wxID_SAVEAS, _T("Save &As..."));
    menuFile->Append(ID_Export, _T("Export..."));

    menuFile->AppendSeparator();
    menuFile->Append(wxID_PRINT, _T("&Print..."));
    menuFile->Append(wxID_PRINT_SETUP, _T("Print &Setup..."));
    menuFile->Append(wxID_PREVIEW, _T("Print Pre&view"));
    
    wxMenu *helpMenu = new wxMenu;
    helpMenu->Append(ID_About, _T("&About"));
    
    wxMenuBar *menuBar = new wxMenuBar;
    
    menuBar->Append(menuFile, _T("&File"));
    menuBar->Append(helpMenu, _T("&Help"));
    
    // Associate the menu bar with the frame
    SetMenuBar(menuBar);

    x_length_box = new wxTextCtrl(this, -1, get_document()->x_length, 
                          wxDefaultPosition, wxDefaultSize);
    y_length_box = new wxTextCtrl(this, -1, get_document()->y_length, 
                          wxDefaultPosition, wxDefaultSize);
    x_phase_box = new wxTextCtrl(this, -1, get_document()->x_phase, 
                          wxDefaultPosition, wxDefaultSize);
    y_phase_box = new wxTextCtrl(this, -1, get_document()->y_phase, 
                          wxDefaultPosition, wxDefaultSize);
    x_amplitude_box = new wxTextCtrl(this, -1, get_document()->x_amplitude, 
                          wxDefaultPosition, wxDefaultSize);
    y_amplitude_box = new wxTextCtrl(this, -1, get_document()->y_amplitude, 
                          wxDefaultPosition, wxDefaultSize);
    friction_box = new wxTextCtrl(this, -1, get_document()->friction, 
                          wxDefaultPosition, wxDefaultSize);
    wxStaticText *x_length_lbl = new wxStaticText(this, -1, "hor length",
                                         wxDefaultPosition, wxDefaultSize);
    wxStaticText *y_length_lbl = new wxStaticText(this, -1, "vert length",
                                         wxDefaultPosition, wxDefaultSize);
    wxStaticText *x_phase_lbl = new wxStaticText(this, -1, "hor phase",
                                         wxDefaultPosition, wxDefaultSize);
    wxStaticText *y_phase_lbl = new wxStaticText(this, -1, "vert phase",
                                         wxDefaultPosition, wxDefaultSize);
    wxStaticText *x_amplitude_lbl = new wxStaticText(this, -1, "hor amplitude",
                                         wxDefaultPosition, wxDefaultSize);
    wxStaticText *y_amplitude_lbl = new wxStaticText(this, -1, "vert amplitude",
                                         wxDefaultPosition, wxDefaultSize);
    wxStaticText *friction_lbl = new wxStaticText(this, -1, "friction",
                                         wxDefaultPosition, wxDefaultSize);
    start_button = new wxButton(this, ID_Start_Button, "Start",
			        wxDefaultPosition, wxDefaultSize);
    stop_button = new wxButton(this, ID_Stop_Button, "Stop",
			        wxDefaultPosition, wxDefaultSize);
    back_button = new wxButton(this, ID_Back_Button, "Back",
			        wxDefaultPosition, wxDefaultSize);

    m_canvas = new Canvas( this, -1, wxDefaultPosition, wxDefaultSize );

    wxBoxSizer *main_sizer = new wxBoxSizer( wxHORIZONTAL );
    wxBoxSizer *control_sizer = new wxBoxSizer(wxVERTICAL);

    main_sizer->Add( control_sizer, 1, wxEXPAND );
    main_sizer->Add( m_canvas, 2, wxEXPAND );

    wxFlexGridSizer *parameter_sizer = new wxFlexGridSizer(7, 2, 1, 1);
    control_sizer->Add( parameter_sizer, 0, wxEXPAND );
    wxBoxSizer *back_sizer = new wxBoxSizer(wxHORIZONTAL);
    control_sizer->Add(back_sizer, 1, wxEXPAND);
    wxBoxSizer *start_sizer = new wxBoxSizer(wxHORIZONTAL);
    control_sizer->Add(start_sizer, 1, wxEXPAND);

    parameter_sizer->AddGrowableCol(1);
    parameter_sizer->Add( x_length_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( x_length_box, 1, wxALL|wxEXPAND );
    parameter_sizer->Add( x_phase_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( x_phase_box, 1, wxALL|wxEXPAND );
    parameter_sizer->Add( x_amplitude_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( x_amplitude_box, 1, wxALL|wxEXPAND );
    parameter_sizer->Add( y_length_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( y_length_box, 1, wxALL|wxEXPAND );
    parameter_sizer->Add( y_phase_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( y_phase_box, 1, wxALL|wxEXPAND );
    parameter_sizer->Add( y_amplitude_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( y_amplitude_box, 1, wxALL|wxEXPAND );
    parameter_sizer->Add( friction_lbl, 0, wxALIGN_CENTER );
    parameter_sizer->Add( friction_box, 1, wxALL|wxEXPAND );

    wxBoxSizer *left_of_back = new wxBoxSizer(wxVERTICAL);
    back_sizer->Add(left_of_back, 1, 0);
    back_sizer->Add(back_button, 0, wxALIGN_BOTTOM);
    wxBoxSizer *right_of_back = new wxBoxSizer(wxVERTICAL);
    back_sizer->Add(right_of_back, 1, 0);

    wxBoxSizer *left_of_start = new wxBoxSizer(wxVERTICAL);
    start_sizer->Add(left_of_start, 1, 0);
    start_sizer->Add(start_button, 0, wxALIGN_BOTTOM);
    wxBoxSizer *right_of_start = new wxBoxSizer(wxVERTICAL);
    start_sizer->Add(right_of_start, 1, 0);
    start_sizer->Add(stop_button, 0, wxALIGN_BOTTOM);
    wxBoxSizer *right_of_stop = new wxBoxSizer(wxVERTICAL);
    start_sizer->Add(right_of_stop, 1, 0);

    SetAutoLayout( TRUE );
    SetSizer( main_sizer );
    
    Centre(wxBOTH);

    enable_moving(false);
    enable_motion_history(false);
}

void Frame::OnQuit(wxCommandEvent& WXUNUSED(event))
{
   timer.Stop();
   Close(TRUE);
}

void Frame::OnExport(wxCommandEvent& WXUNUSED(event))
{
   // User selects a size.
   wxString choices[] = {"256","512","768", "1024", "2048","4096"};
   int choice_vals[] = {256,512,768,1024,2048,4096};
   wxSingleChoiceDialog dlg(this, "Choose size in pixels.", "Size", 6, choices);
   dlg.ShowModal();
   int size = choice_vals[dlg.GetSelection()];

   // Create the bitmap.
   wxBitmap bitmap(size, size, 1 /* monochrome */);
   wxMemoryDC dc;
   dc.SelectObject(bitmap);
   m_view->OnDraw(&dc);
   dc.SelectObject(wxNullBitmap);

   // User selects a file.
#if 0
   wxList handlers = bitmap.GetHandlers();
   for (int i = 0; i < handlers.GetCount(); i += 1) {
       wxBitmapHandler *hp = handlers.Item(i).GetData();
       printf("extension %s\n", hp->GetExtension().c_str());
   }
#endif
#ifdef __WXMSW__
   wxFileDialog fdlg(this, "Choose a file for export.", "", "", "BMP files (*.bmp)|*.bmp");
   if (fdlg.ShowModal() == wxID_OK) {
       bitmap.SaveFile(fdlg.GetPath(), wxBITMAP_TYPE_BMP);
   }
#endif
#ifdef __WXGTK__
   wxFileDialog fdlg(this, "Choose a file for export.", "", "", "XPM files (*.xpm)|*.xpm");
   if (fdlg.ShowModal() == wxID_OK) {
       bitmap.SaveFile(fdlg.GetPath(), wxBITMAP_TYPE_XPM);
   }
#endif
}

void Frame::OnTimer(wxTimerEvent &event)
{
    TrialDocument &document = *get_document();
    Pendulums *pendulums = document.pendulums;
    if (pendulums != 0) {
	PendulumsState ps0 = pendulums->get_state();
	bool advanced = pendulums->advance(double(event.GetInterval())/1000.0);
	if (advanced) {
	    document.Modify(true);
	    PendulumsState ps1 = pendulums->get_state();
            m_canvas->tick(ps0, ps1);
        }
        else {
            timer.Stop();
            stop_button->Enable(false);
        }
    }
}

void Frame::update_parameters(TrialDocument &document) {
    if (document.y_phase != y_phase_box->GetValue()
	|| document.x_phase != x_phase_box->GetValue()
	|| document.x_length != x_length_box->GetValue()
	|| document.y_length != y_length_box->GetValue()
	|| document.x_amplitude != x_amplitude_box->GetValue()
	|| document.y_amplitude != y_amplitude_box->GetValue()
	|| document.friction != friction_box->GetValue()) {
	document.Modify(true);
    }

    document.y_phase = y_phase_box->GetValue();
    document.x_phase = x_phase_box->GetValue();
    document.x_length = x_length_box->GetValue();
    document.y_length = y_length_box->GetValue();
    document.x_amplitude = x_amplitude_box->GetValue();
    document.y_amplitude = y_amplitude_box->GetValue();
    document.friction = friction_box->GetValue();
}

void Frame::update_parameter_boxes(TrialDocument &document) {
    y_phase_box->SetValue(document.y_phase);
    x_phase_box->SetValue(document.x_phase);
    x_length_box->SetValue(document.x_length);
    y_length_box->SetValue(document.y_length);
    x_amplitude_box->SetValue(document.x_amplitude);
    y_amplitude_box->SetValue(document.y_amplitude);
    friction_box->SetValue(document.friction);
}

void Frame::refresh() {
#ifdef __WXMSW__
     // Need for this to be fixed in wxwidgets 2.5.1.
     m_canvas->Refresh();
#endif
     Refresh();
}

void Frame::OnStart(wxCommandEvent& WXUNUSED(event)) {
    TrialDocument &document = *get_document();
    if (document.pendulums == NULL) {
	wxString diagnostic;
	update_parameters(document);
	if (document.set_pendulums(diagnostic)) {
            document.Modify(true);
	}
	else {
	    wxMessageDialog diag(this, diagnostic, "Error", wxOK|wxICON_ERROR);
	    diag.ShowModal();
	}
    }
    else {
        document.pendulums->pause(false);
    }
    if (document.pendulums != NULL) {
        enable_motion_history(true);
        enable_moving(true);
	timer.Start(12);
    }
}

void Frame::OnStop(wxCommandEvent& WXUNUSED(event)) {
    Pendulums *pendulums = get_document()->pendulums;
    if (pendulums != 0) {
	pendulums->pause(true);
    }
    timer.Stop();
    enable_moving(false);
}

void Frame::OnBack(wxCommandEvent& WXUNUSED(event)) {
    timer.Stop();
    get_document()->cancel_motion();
    enable_moving(false);
    enable_motion_history(false);
}

TrialDocument *Frame::get_document() {
    return wxDynamicCast(GetDocument(), TrialDocument);
}

void Frame::enable_moving(bool is_moving) {
    // Stop only when running.
    stop_button->Enable(is_moving);
    // Start only when not running.
    start_button->Enable(!is_moving);
}

void Frame::enable_motion_history(bool has_moved) {
    // Print and export menu items only when pendulums exist.
    menuFile->Enable(wxID_PRINT, has_moved);
    menuFile->Enable(wxID_PREVIEW, has_moved);
    menuFile->Enable(ID_Export, has_moved);

    // Parameters editable only when pendulums do not exist.
    x_length_box->Enable(!has_moved);
    y_length_box->Enable(!has_moved);
    x_phase_box->Enable(!has_moved);
    y_phase_box->Enable(!has_moved);
    x_amplitude_box->Enable(!has_moved);
    y_amplitude_box->Enable(!has_moved);
    friction_box->Enable(!has_moved);

    // Back only when pendulums exist.
    back_button->Enable(has_moved);
}
