/*
  harmonograph is Copyright (c) Prescott K. Turner, 2006. All rights reserved.
  It is distributed as free software under the license in the file "License",
  which is included in the distribution.
*/
#include "wx/wx.h"

#include "harmonograph.h"
#include "trialdoc.h"
#include "trialview.h"
#include "pendulums.h"
#include "docmanager.h"
#include "frame.h"
#include <list>

#include <unistd.h>

using namespace std;

IMPLEMENT_DYNAMIC_CLASS(TrialView, wxView)

TrialView::TrialView() { frame = NULL; };
TrialView::~TrialView() {}

// What to do when a view is created. Creates actual
// windows for displaying the view.
bool TrialView::OnCreate(wxDocument *doc, long WXUNUSED(flags) )
{
    wxFrame *main_frame = wxDynamicCast(GetDocumentManager(), DocManager)->main_frame;
    frame = new Frame(doc, this, main_frame, _T("TrialView"), wxPoint(10, 10), wxSize(600, 420));

#ifdef __X__
    // X seems to require a forced resize
    int x, y;
    frame->GetSize(&x, &y);
    frame->SetSize(-1, -1, x, y);
#endif
    frame->Show(TRUE);
    return TRUE;
}

void size_DC(wxDC &dc, double &x_scale, double &y_scale)
{
    int size_x = 0;
    int size_y = 0;
    dc.GetSize( &size_x, &size_y );

    if (size_x < size_y) {
	y_scale = x_scale = double(size_x-1)/2.0;
    }
    else {
	y_scale = x_scale = double(size_y-1)/2.0;
    }
}

void TrialView::segment(wxDC &dc, const PendulumsState &ps0, const PendulumsState &ps1)
{
    double x_scale, y_scale;
    size_DC(dc, x_scale, y_scale);
    presized_segment(dc, ps0, ps1, x_scale, y_scale);
}

void TrialView::presized_segment(wxDC &dc, const PendulumsState &ps0, const PendulumsState &ps1, double x_scale, double y_scale)
{
    dc.SetPen( *wxBLACK_PEN );

    int prev_x = round2i((ps0.x.position+1.0)*x_scale);
    int prev_y = round2i((ps0.y.position+1.0)*y_scale);
    int x = round2i((ps1.x.position+1.0)*x_scale);
    int y = round2i((ps1.y.position+1.0)*y_scale);

    // dc.DrawLine(prev_x, prev_y,prev_x,prev_y);
    dc.DrawLine(prev_x,prev_y,x,y);
}

// Sneakily gets used for default print/preview
// as well as drawing on the screen.
void TrialView::OnDraw(wxDC *dc)
{
    dc->SetPen(*wxBLACK_PEN);
    dc->SetMapMode(wxMM_TEXT);
    dc->SetUserScale(1.0, 1.0);

    double x_scale, y_scale;
    size_DC(*dc, x_scale, y_scale);

    dc->Clear();

    list<PendulumsState>::reverse_iterator iter;
    TrialDocument *document = wxDynamicCast(GetDocument(), TrialDocument);
    if (document != NULL) {
	list<PendulumsState> &history = document->get_pendulums_history();
	PendulumsState *ps0 = 0;
	for (iter = history.rbegin(); iter != history.rend(); iter++) {
	    PendulumsState &ps1 = *iter;
	    if (ps0 != 0) {
		presized_segment(*dc, *ps0, ps1, x_scale, y_scale);
	    }
	    ps0 = &ps1;
	}
	if (ps0 != 0) {
	    PendulumsState ps1 = document->get_pendulums_state();
	    presized_segment(*dc, *ps0, ps1, x_scale, y_scale);
	}
    }
}

void TrialView::OnUpdate(wxView *WXUNUSED(sender), wxObject *WXUNUSED(hint))
{
    if (frame != 0) {
        TrialDocument &document = *(wxDynamicCast(GetDocument(), TrialDocument));
	frame->update_parameter_boxes(document);
	frame->enable_motion_history(document.pendulums != NULL);
	frame->enable_moving(frame->timer.IsRunning());
        frame->refresh();
    }
}

// Clean up windows used for displaying the view.
bool TrialView::OnClose(bool deleteWindow)
{
    if (!GetDocument()->Close())
        return FALSE;
    
    wxString s(wxTheApp->GetAppName());
    if (frame != 0)
        frame->SetTitle(s);
    
    SetFrame(NULL);
    
    Activate(FALSE);
    
    if (deleteWindow)
    {
        delete frame;
	frame = NULL;
        return TRUE;
    }
    frame->m_view = NULL;
    return TRUE;
}
